<?php

namespace App\Http\Controllers;

use App\Models\AgendaEvento;
use App\Models\AgendaEventoParticipante;
use App\Models\Cliente;
use App\Models\Usuario;
use App\Services\AgendaService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AgendaController extends Controller
{
    private const CATEGORIAS = ['reuniao', 'tarefa', 'lembrete', 'prazo'];
    private const STATUS = ['pendente', 'em_andamento', 'concluido', 'cancelado'];
    private const PRIORIDADES = ['alta', 'media', 'baixa'];
    private const VISIBILIDADES = ['privado', 'compartilhado', 'global'];

    public function __construct(private AgendaService $service)
    {
    }

    public function index(Request $request)
    {
        $user = $request->user();
        $query = AgendaEvento::query()
            ->with([
                'organizador:id,name,email',
                'criadoPor:id,name,email',
                'atualizadoPor:id,name,email',
                'participantes.usuario:id,name,email',
            ]);

        $isAdmin = $this->service->isAdmin($user);
        if (!$isAdmin) {
            $query->where(function ($builder) use ($user) {
                $builder->where('owner_id', $user->id)
                    ->orWhere('visibilidade', 'global')
                    ->orWhereHas('participantes', function ($sub) use ($user) {
                        $sub->where('usuario_id', $user->id);
                    });
            });
        }

        $start = $request->query('start');
        $end = $request->query('end');
        if ($start) {
            $startAt = Carbon::parse($start)->startOfDay();
            $query->where('fim_em', '>=', $startAt);
        }
        if ($end) {
            $endAt = Carbon::parse($end)->endOfDay();
            $query->where('inicio_em', '<=', $endAt);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->string('status'));
        }
        if ($request->filled('categoria')) {
            $query->where('categoria', $request->string('categoria'));
        }
        if ($request->filled('prioridade')) {
            $query->where('prioridade', $request->string('prioridade'));
        }
        if ($request->filled('visibilidade')) {
            $query->where('visibilidade', $request->string('visibilidade'));
        }
        if ($request->filled('tag')) {
            $tag = $request->string('tag')->toString();
            $query->whereJsonContains('tags', $tag);
        }
        if ($isAdmin && $request->filled('usuario_id')) {
            $userId = (int) $request->input('usuario_id');
            $query->where(function ($builder) use ($userId) {
                $builder->where('owner_id', $userId)
                    ->orWhereHas('participantes', function ($sub) use ($userId) {
                        $sub->where('usuario_id', $userId);
                    });
            });
        }
        if ($request->filled('search')) {
            $search = '%'.trim($request->input('search')).'%';
            $query->where(function ($builder) use ($search) {
                $builder->where('titulo', 'like', $search)
                    ->orWhere('descricao', 'like', $search)
                    ->orWhere('localizacao', 'like', $search)
                    ->orWhere('cliente_nome', 'like', $search);
            });
        }

        $limit = min((int) $request->query('limit', 500), 1000);
        $eventos = $query->orderBy('inicio_em')->limit($limit)->get();

        return response()->json([
            'data' => $eventos,
        ]);
    }

    public function show(Request $request, AgendaEvento $agenda)
    {
        $user = $request->user();
        if (!$this->service->canView($user, $agenda)) {
            return response()->json(['message' => 'Sem permissao para visualizar.'], 403);
        }

        $agenda->load([
            'organizador:id,name,email',
            'criadoPor:id,name,email',
            'atualizadoPor:id,name,email',
            'participantes.usuario:id,name,email',
            'mensagens.usuario:id,name',
        ]);

        return response()->json($agenda);
    }

    public function store(Request $request)
    {
        $user = $request->user();
        $data = $this->validatePayload($request, true);

        if (!$this->service->isAdmin($user)) {
            $data['visibilidade'] = $data['visibilidade'] === 'global' ? 'privado' : $data['visibilidade'];
            $data['owner_id'] = $user->id;
        } elseif (empty($data['owner_id'])) {
            $data['owner_id'] = $user->id;
        }

        $data['created_by_id'] = $user->id;
        $data['updated_by_id'] = $user->id;
        $data['tags'] = $this->normalizeStringArray($data['tags'] ?? null);
        $data['relacionados'] = $this->normalizeNumericArray($data['relacionados'] ?? null);

        $agenda = AgendaEvento::query()->create($data);

        $this->syncParticipantes($agenda, $data['participantes'] ?? []);
        $this->fillClienteNome($agenda);
        $agenda->refresh();

        $this->service->syncReminders($agenda);

        return response()->json($agenda->load('participantes.usuario:id,name,email'), 201);
    }

    public function update(Request $request, AgendaEvento $agenda)
    {
        $user = $request->user();
        if (!$this->service->canEdit($user, $agenda)) {
            return response()->json(['message' => 'Sem permissao para editar.'], 403);
        }

        $data = $this->validatePayload($request, false);

        if (!$this->service->isAdmin($user)) {
            unset($data['owner_id']);
            if (isset($data['visibilidade']) && $data['visibilidade'] === 'global') {
                $data['visibilidade'] = $agenda->visibilidade;
            }
        }

        if (array_key_exists('tags', $data)) {
            $data['tags'] = $this->normalizeStringArray($data['tags'] ?? null);
        }
        if (array_key_exists('relacionados', $data)) {
            $data['relacionados'] = $this->normalizeNumericArray($data['relacionados'] ?? null);
        }

        $data['updated_by_id'] = $user->id;
        $agenda->fill($data);
        $agenda->save();

        if (array_key_exists('participantes', $data)) {
            $this->syncParticipantes($agenda, $data['participantes'] ?? []);
        }

        $this->fillClienteNome($agenda);
        $agenda->refresh();

        $this->service->syncReminders($agenda);

        return response()->json($agenda->load('participantes.usuario:id,name,email'));
    }

    public function destroy(Request $request, AgendaEvento $agenda)
    {
        $user = $request->user();
        if (!$this->service->canEdit($user, $agenda)) {
            return response()->json(['message' => 'Sem permissao para excluir.'], 403);
        }

        DB::transaction(function () use ($agenda) {
            $agenda->mensagens()->delete();
            $agenda->participantes()->delete();
            $agenda->lembretes()->delete();
            $agenda->delete();
        });

        return response()->noContent();
    }

    public function stats(Request $request)
    {
        $user = $request->user();
        $query = AgendaEvento::query();
        $isAdmin = $this->service->isAdmin($user);

        if (!$isAdmin) {
            $query->where(function ($builder) use ($user) {
                $builder->where('owner_id', $user->id)
                    ->orWhere('visibilidade', 'global')
                    ->orWhereHas('participantes', function ($sub) use ($user) {
                        $sub->where('usuario_id', $user->id);
                    });
            });
        }

        if ($request->filled('start')) {
            $startAt = Carbon::parse($request->input('start'))->startOfDay();
            $query->where('fim_em', '>=', $startAt);
        }
        if ($request->filled('end')) {
            $endAt = Carbon::parse($request->input('end'))->endOfDay();
            $query->where('inicio_em', '<=', $endAt);
        }

        $items = $query->get(['id', 'inicio_em', 'fim_em', 'categoria', 'status']);

        $total = $items->count();
        $concluidos = $items->where('status', 'concluido')->count();
        $cancelados = $items->where('status', 'cancelado')->count();
        $reunioes = $items->where('categoria', 'reuniao')->count();
        $tarefas = $items->where('categoria', 'tarefa')->count();
        $tempoTotal = 0;
        foreach ($items as $item) {
            if (!$item->inicio_em || !$item->fim_em) {
                continue;
            }
            $tempoTotal += Carbon::parse($item->fim_em)->diffInMinutes(Carbon::parse($item->inicio_em));
        }

        return response()->json([
            'total' => $total,
            'concluidos' => $concluidos,
            'cancelados' => $cancelados,
            'reunioes' => $reunioes,
            'tarefas' => $tarefas,
            'tempo_total_minutos' => $tempoTotal,
        ]);
    }

    public function export(Request $request)
    {
        $request->merge(['limit' => 1000]);
        $response = $this->index($request)->getData(true);
        $rows = $response['data'] ?? [];

        $csv = fopen('php://temp', 'r+');
        fputcsv($csv, [
            'ID',
            'Titulo',
            'Inicio',
            'Fim',
            'Categoria',
            'Status',
            'Prioridade',
            'Visibilidade',
            'Organizador',
            'Cliente',
            'Localizacao',
        ]);
        foreach ($rows as $row) {
            fputcsv($csv, [
                $row['id'] ?? '',
                $row['titulo'] ?? '',
                $row['inicio_em'] ?? '',
                $row['fim_em'] ?? '',
                $row['categoria'] ?? '',
                $row['status'] ?? '',
                $row['prioridade'] ?? '',
                $row['visibilidade'] ?? '',
                $row['organizador']['name'] ?? '',
                $row['cliente_nome'] ?? '',
                $row['localizacao'] ?? '',
            ]);
        }
        rewind($csv);
        $content = stream_get_contents($csv);
        fclose($csv);

        return response($content, 200, [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename=\"agenda.csv\"',
        ]);
    }

    public function usuarios(Request $request)
    {
        $search = trim((string) $request->query('search', ''));

        $query = Usuario::query()
            ->select(['id', 'name', 'email', 'role', 'ativo'])
            ->where('ativo', true)
            ->orderBy('name');

        if ($search !== '') {
            $query->where(function ($builder) use ($search) {
                $builder->where('name', 'like', '%'.$search.'%')
                    ->orWhere('email', 'like', '%'.$search.'%');
            });
        }

        return response()->json($query->limit(50)->get());
    }

    public function clientes(Request $request)
    {
        $search = trim((string) $request->query('search', ''));

        $query = Cliente::query()->select(['id', 'nome_razao', 'cpf_cnpj']);
        if ($search !== '') {
            $query->where(function ($builder) use ($search) {
                $builder->where('nome_razao', 'like', '%'.$search.'%')
                    ->orWhere('cpf_cnpj', 'like', '%'.$search.'%');
            });
        }

        return response()->json($query->orderBy('nome_razao')->limit(50)->get());
    }

    private function validatePayload(Request $request, bool $creating): array
    {
        $rules = [
            'titulo' => ['required', 'string', 'max:255'],
            'descricao' => ['nullable', 'string'],
            'inicio_em' => ['required', 'date'],
            'fim_em' => ['required', 'date', 'after_or_equal:inicio_em'],
            'categoria' => ['nullable', 'string', 'in:'.implode(',', self::CATEGORIAS)],
            'status' => ['nullable', 'string', 'in:'.implode(',', self::STATUS)],
            'prioridade' => ['nullable', 'string', 'in:'.implode(',', self::PRIORIDADES)],
            'localizacao' => ['nullable', 'string', 'max:255'],
            'notas' => ['nullable', 'string'],
            'tags' => ['nullable', 'array'],
            'tags.*' => ['string', 'max:60'],
            'relacionados' => ['nullable', 'array'],
            'relacionados.*' => ['integer'],
            'visibilidade' => ['nullable', 'string', 'in:'.implode(',', self::VISIBILIDADES)],
            'owner_id' => ['nullable', 'integer', 'exists:users,id'],
            'usuario_telefone' => ['nullable', 'string', 'max:40'],
            'cliente_id' => ['nullable', 'integer', 'exists:clientes,id'],
            'cliente_nome' => ['nullable', 'string', 'max:255'],
            'cliente_telefone' => ['nullable', 'string', 'max:40'],
            'cliente_email' => ['nullable', 'email', 'max:255'],
            'lembrete_minutos' => ['nullable', 'integer', 'min:0', 'max:10080'],
            'lembrete_sms_usuario' => ['nullable', 'boolean'],
            'lembrete_sms_cliente' => ['nullable', 'boolean'],
            'participantes' => ['nullable', 'array'],
            'participantes.*' => ['integer', 'exists:users,id'],
        ];

        if (!$creating) {
            foreach ($rules as $key => $rule) {
                $rules[$key] = array_merge(['sometimes'], (array) $rule);
            }
        }

        return $request->validate($rules);
    }

    private function normalizeStringArray($value): ?array
    {
        if ($value === null) {
            return null;
        }
        $items = is_array($value) ? $value : explode(',', (string) $value);
        $filtered = array_values(array_filter(array_map('trim', $items), fn ($item) => $item !== ''));
        return $filtered ? array_values(array_unique($filtered)) : null;
    }

    private function normalizeNumericArray($value): ?array
    {
        if ($value === null) {
            return null;
        }
        $items = is_array($value) ? $value : explode(',', (string) $value);
        $filtered = [];
        foreach ($items as $item) {
            $id = (int) $item;
            if ($id > 0) {
                $filtered[] = $id;
            }
        }
        return $filtered ? array_values(array_unique($filtered)) : null;
    }

    private function syncParticipantes(AgendaEvento $agenda, array $participantes): void
    {
        $ids = array_values(array_unique(array_map('intval', $participantes)));
        $ids = array_filter($ids, fn ($id) => $id > 0 && $id !== (int) $agenda->owner_id);

        $existing = AgendaEventoParticipante::query()
            ->where('evento_id', $agenda->id)
            ->pluck('usuario_id')
            ->all();

        $toAdd = array_diff($ids, $existing);
        $toRemove = array_diff($existing, $ids);

        if ($toRemove) {
            AgendaEventoParticipante::query()
                ->where('evento_id', $agenda->id)
                ->whereIn('usuario_id', $toRemove)
                ->delete();
        }

        foreach ($toAdd as $id) {
            AgendaEventoParticipante::query()->create([
                'evento_id' => $agenda->id,
                'usuario_id' => $id,
                'status' => 'pendente',
            ]);
        }
    }

    private function fillClienteNome(AgendaEvento $agenda): void
    {
        if (!$agenda->cliente_id || $agenda->cliente_nome) {
            return;
        }

        $cliente = Cliente::query()->find($agenda->cliente_id);
        if (!$cliente) {
            return;
        }

        $agenda->cliente_nome = $cliente->nome_razao;
        $agenda->save();
    }
}
