<?php

namespace App\Http\Controllers;

use App\Models\Configuracao;
use Illuminate\Http\Request;

class ConfiguracoesController extends Controller
{
    private const DEFAULT_SMS_TEXTO_PROPOSTA = "LPC Capital - documentos da proposta\nOla {nome}\nObjetivo: {objetivo}\nDocumentos: {documentos}\nEnvio: {link}";
    private const DEFAULT_SMS_TEXTO_CONTRATO = "LPC Capital - documentos do contrato\nOla {nome}\nObjetivo: {objetivo}\nDocumentos: {documentos}\nEnvio: {link}";
    private const DEFAULT_SMS_TEXTO_AGENDA_USUARIO = "LPC Capital - lembrete de agenda\nOla {responsavel}\nEvento: {titulo}\nInicio: {inicio}\nLocal: {local}\nCliente: {cliente}";
    private const DEFAULT_SMS_TEXTO_AGENDA_CLIENTE = "LPC Capital - lembrete de agenda\nOla {cliente}\nCompromisso: {titulo}\nInicio: {inicio}\nLocal: {local}\nResponsavel: {responsavel}";

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->requireRole(['admin']);
            return $next($request);
        });
    }

    public function index()
    {
        return response()->json($this->formatConfig(true));
    }

    public function update(Request $request)
    {
        $data = $this->validateConfig($request);
        $config = $this->loadConfig();

        $payload = [
            'whatsapp_api_url' => $this->normalizeValue($data['whatsapp_api_url'] ?? null),
            'whatsapp_api_token' => $this->resolveSecret(
                $data['whatsapp_api_token'] ?? null,
                $config->whatsapp_api_token
            ),
            'whatsapp_api_key' => $this->resolveSecret(
                $data['whatsapp_api_key'] ?? null,
                $config->whatsapp_api_key
            ),
            'sms_api_url' => $this->normalizeValue($data['sms_api_url'] ?? null),
            'sms_auth_key' => $this->resolveSecret(
                $data['sms_auth_key'] ?? null,
                $config->sms_auth_key
            ),
            'sms_sender' => $this->normalizeValue($data['sms_sender'] ?? null),
            'sms_sender_contrato' => $this->normalizeValue($data['sms_sender_contrato'] ?? null),
            'sms_texto_proposta' => $this->normalizeValue($data['sms_texto_proposta'] ?? null),
            'sms_texto_contrato' => $this->normalizeValue($data['sms_texto_contrato'] ?? null),
            'sms_texto_agenda_usuario' => $this->normalizeValue($data['sms_texto_agenda_usuario'] ?? null),
            'sms_texto_agenda_cliente' => $this->normalizeValue($data['sms_texto_agenda_cliente'] ?? null),
        ];

        $config->fill($payload);
        $config->save();

        return response()->json($this->formatConfig(true, $config));
    }

    public function showWhatsapp()
    {
        return response()->json($this->resolveWhatsappConfig(true));
    }

    public function updateWhatsapp(Request $request)
    {
        $data = $request->validate([
            'api_url' => ['nullable', 'string', 'max:255'],
            'api_token' => ['nullable', 'string', 'max:255'],
            'api_key' => ['nullable', 'string', 'max:255'],
        ]);

        $config = $this->loadConfig();
        $token = $this->resolveSecret($data['api_token'] ?? null, $config->whatsapp_api_token);
        $key = $this->resolveSecret($data['api_key'] ?? null, $config->whatsapp_api_key);

        $payload = [
            'whatsapp_api_url' => $this->normalizeValue($data['api_url'] ?? null),
            'whatsapp_api_token' => $token,
            'whatsapp_api_key' => $key,
        ];

        $config->fill($payload);
        $config->save();

        return response()->json([
            'message' => 'Configuracoes atualizadas.',
            'data' => $this->resolveWhatsappConfig(true, $config),
        ]);
    }

    private function loadConfig(): Configuracao
    {
        $config = Configuracao::query()->first();
        if (!$config) {
            $config = Configuracao::query()->create();
        }
        return $config;
    }

    private function validateConfig(Request $request): array
    {
        return $request->validate([
            'whatsapp_api_url' => ['nullable', 'string', 'max:255'],
            'whatsapp_api_token' => ['nullable', 'string', 'max:255'],
            'whatsapp_api_key' => ['nullable', 'string', 'max:255'],
            'sms_api_url' => ['nullable', 'string', 'max:255'],
            'sms_auth_key' => ['nullable', 'string', 'max:255'],
            'sms_sender' => ['nullable', 'string', 'max:255'],
            'sms_sender_contrato' => ['nullable', 'string', 'max:255'],
            'sms_texto_proposta' => ['nullable', 'string', 'max:2000'],
            'sms_texto_contrato' => ['nullable', 'string', 'max:2000'],
            'sms_texto_agenda_usuario' => ['nullable', 'string', 'max:2000'],
            'sms_texto_agenda_cliente' => ['nullable', 'string', 'max:2000'],
        ]);
    }

    private function formatConfig(bool $masked = false, ?Configuracao $config = null): array
    {
        return array_merge(
            $this->resolveWhatsappConfig($masked, $config),
            $this->resolveSmsConfig($masked, $config)
        );
    }

    private function resolveWhatsappConfig(bool $masked = false, ?Configuracao $config = null): array
    {
        $config = $config ?: Configuracao::query()->first();
        $token = $config?->whatsapp_api_token ?: env('WHATSAPP_API_TOKEN');
        $key = $config?->whatsapp_api_key ?: env('WHATSAPP_API_KEY');

        return [
            'whatsapp_api_url' => $config?->whatsapp_api_url ?: env('WHATSAPP_API_URL'),
            'whatsapp_api_token' => $masked ? $this->maskSecret($token) : $token,
            'whatsapp_api_key' => $masked ? $this->maskSecret($key) : $key,
        ];
    }

    private function resolveSmsConfig(bool $masked = false, ?Configuracao $config = null): array
    {
        $config = $config ?: Configuracao::query()->first();
        $authKey = $config?->sms_auth_key ?: env('COMTELE_SMS_AUTH_KEY');
        $defaultUrl = 'https://sms.comtele.com.br/api/v2/send';

        return [
            'sms_api_url' => $config?->sms_api_url ?: env('COMTELE_SMS_API_URL', $defaultUrl),
            'sms_auth_key' => $masked ? $this->maskSecret($authKey) : $authKey,
            'sms_sender' => $config?->sms_sender ?: env('COMTELE_SMS_SENDER'),
            'sms_sender_contrato' => $config?->sms_sender_contrato ?: env('COMTELE_SMS_SENDER_CONTRATO'),
            'sms_texto_proposta' => $config?->sms_texto_proposta
                ?: env('COMTELE_SMS_TEXTO_PROPOSTA', self::DEFAULT_SMS_TEXTO_PROPOSTA),
            'sms_texto_contrato' => $config?->sms_texto_contrato
                ?: env('COMTELE_SMS_TEXTO_CONTRATO', self::DEFAULT_SMS_TEXTO_CONTRATO),
            'sms_texto_agenda_usuario' => $config?->sms_texto_agenda_usuario
                ?: env('COMTELE_SMS_TEXTO_AGENDA_USUARIO', self::DEFAULT_SMS_TEXTO_AGENDA_USUARIO),
            'sms_texto_agenda_cliente' => $config?->sms_texto_agenda_cliente
                ?: env('COMTELE_SMS_TEXTO_AGENDA_CLIENTE', self::DEFAULT_SMS_TEXTO_AGENDA_CLIENTE),
        ];
    }

    private function normalizeValue(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }
        $text = trim($value);
        return $text === '' ? null : $text;
    }

    private function maskSecret(?string $value): ?string
    {
        if ($value === null || $value === '') {
            return null;
        }
        $length = strlen($value);
        if ($length <= 4) {
            return str_repeat('*', $length);
        }
        return str_repeat('*', $length - 4).substr($value, -4);
    }

    private function isMaskedSecret(?string $value): bool
    {
        if ($value === null || $value === '') {
            return false;
        }
        return str_contains($value, '*');
    }

    private function resolveSecret(?string $incoming, ?string $current): ?string
    {
        $value = $this->normalizeValue($incoming);
        if ($this->isMaskedSecret($value)) {
            return $current;
        }
        return $value;
    }
}
