<?php

namespace App\Http\Controllers;

use App\Models\Parceiro;
use Illuminate\Http\Request;

class ParceiroController extends Controller
{
    private const TIPOS = ['banco', 'parceiro'];
    private const CANAIS = ['api', 'email'];

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->requireRole(['admin', 'gestor']);
            return $next($request);
        });
    }

    public function index()
    {
        $parceiros = Parceiro::query()
            ->orderByDesc('id')
            ->paginate(20);

        return response()->json($parceiros);
    }

    public function create()
    {
        return response()->json(['message' => 'create']);
    }

    public function store(Request $request)
    {
        $data = $this->validatePayload($request);

        if (!array_key_exists('ativo', $data)) {
            $data['ativo'] = true;
        }

        $parceiro = Parceiro::query()->create($data);

        return response()->json($parceiro, 201);
    }

    public function show(int $id)
    {
        return response()->json(Parceiro::query()->findOrFail($id));
    }

    public function edit(int $id)
    {
        return response()->json(Parceiro::query()->findOrFail($id));
    }

    public function update(Request $request, int $id)
    {
        $parceiro = Parceiro::query()->findOrFail($id);
        $data = $this->validatePayload($request, false);
        $parceiro->update($data);

        return response()->json($parceiro);
    }

    public function destroy(int $id)
    {
        Parceiro::query()->whereKey($id)->delete();
        return response()->json(['message' => 'Parceiro removido.']);
    }

    private function validatePayload(Request $request, bool $isCreate = true): array
    {
        $required = $isCreate ? 'required' : 'sometimes';
        return $request->validate([
            'nome' => [$required, 'string', 'max:160'],
            'tipo' => ['nullable', 'string', 'in:'.implode(',', self::TIPOS)],
            'canal_envio' => ['nullable', 'string', 'in:'.implode(',', self::CANAIS)],
            'contato_nome' => ['nullable', 'string', 'max:120'],
            'contato_email' => ['nullable', 'email', 'max:160'],
            'contato_telefone' => ['nullable', 'string', 'max:40'],
            'api_url' => ['nullable', 'string', 'max:255'],
            'api_token' => ['nullable', 'string', 'max:255'],
            'api_key' => ['nullable', 'string', 'max:255'],
            'api_headers' => ['nullable', 'string'],
            'ativo' => ['nullable', 'boolean'],
        ]);
    }
}
