<?php

namespace App\Http\Controllers;

use App\Models\Cliente;
use App\Models\Lead;
use App\Models\Parceiro;
use App\Models\PropostaCredito;
use App\Models\PropostaDocumento;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Schema;

class PropostaCreditoController extends Controller
{
    private const EXTERNAL_DB = 'ipccapitalmarian_lpc_simulacao';

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->requireRole(['admin', 'gestor']);
            return $next($request);
        });
    }

    public function index()
    {
        $propostas = PropostaCredito::query()
            ->with('cliente')
            ->orderByDesc('id')
            ->paginate(20);

        return response()->json($propostas);
    }

    public function create()
    {
        return response()->json(['message' => 'create']);
    }

    public function store(Request $request)
    {
        $registro = PropostaCredito::query()->create($request->only([
            'cliente_id',
            'valor_solicitado',
            'prazo_meses',
            'finalidade',
            'status_proposta_id',
        ]));

        return response()->json($registro->load('cliente'), 201);
    }

    public function show(int $id)
    {
        $proposta = PropostaCredito::query()
            ->with(['cliente', 'propostaDocumentos', 'solicitacoesDocumentos'])
            ->findOrFail($id);

        $payload = $proposta->toArray();
        $payload['documentos'] = $this->mapDocumentos($proposta->propostaDocumentos);
        $payload['solicitacoes_documentos'] = $this->mapSolicitacoes($proposta->solicitacoesDocumentos);

        return response()->json($payload);
    }

    public function edit(int $id)
    {
        return response()->json(PropostaCredito::query()->findOrFail($id));
    }

    public function update(Request $request, int $id)
    {
        $registro = PropostaCredito::query()->findOrFail($id);
        $registro->update($request->only([
            'valor_solicitado',
            'prazo_meses',
            'finalidade',
            'status_proposta_id',
        ]));

        return response()->json($registro->load('cliente'));
    }

    public function destroy(int $id)
    {
        PropostaCredito::query()->whereKey($id)->delete();
        return response()->noContent();
    }

    public function suggest(int $clienteId)
    {
        $cliente = Cliente::query()->find($clienteId);
        if (!$cliente) {
            return response()->json(['message' => 'Cliente nao encontrado.'], 404);
        }

        $defaultStatusId = $this->resolveDefaultStatusId();

        $proposta = PropostaCredito::query()
            ->where('cliente_id', $clienteId)
            ->orderByDesc('id')
            ->first();

        if ($proposta) {
            return response()->json([
                'source' => 'proposta',
                'valor_solicitado' => $proposta->valor_solicitado,
                'prazo_meses' => $proposta->prazo_meses,
                'finalidade' => $proposta->finalidade,
                'status_proposta_id' => $proposta->status_proposta_id ?? $defaultStatusId,
            ]);
        }

        $lead = Lead::query()
            ->where(function ($query) use ($clienteId) {
                $query->where('pessoa_fisica_id', $clienteId)
                    ->orWhere('pessoa_juridica_id', $clienteId);
            })
            ->orderByDesc('id')
            ->first();

        $external = $this->loadExternalSuggestion($cliente, $lead);
        if ($external) {
            return response()->json([
                'source' => 'lead',
                'valor_solicitado' => $external->valor_emprestimo ?? null,
                'prazo_meses' => $external->prazo ?? null,
                'finalidade' => $lead?->tipo_solicitacao ?? null,
                'status_proposta_id' => $defaultStatusId,
            ]);
        }

        return response()->json([
            'source' => null,
            'valor_solicitado' => null,
            'prazo_meses' => null,
            'finalidade' => null,
            'status_proposta_id' => $defaultStatusId,
        ]);
    }

    public function enviarParceiro(Request $request, PropostaCredito $proposta)
    {
        $payload = $request->validate([
            'parceiro_id' => ['required', 'integer', 'exists:parceiros,id'],
            'canal' => ['nullable', 'string', 'in:api,email'],
        ]);

        $parceiro = Parceiro::query()->findOrFail($payload['parceiro_id']);
        if (!$parceiro->ativo) {
            return response()->json(['message' => 'Parceiro inativo.'], 422);
        }

        $canal = $payload['canal'] ?: ($parceiro->canal_envio ?: 'email');
        $data = $this->buildParceiroPayload($proposta);

        if ($canal === 'api') {
            if (!$parceiro->api_url) {
                return response()->json(['message' => 'API do parceiro nao configurada.'], 422);
            }
            $result = $this->sendParceiroApi($parceiro, $data);
            if (!$result['ok']) {
                return response()->json([
                    'message' => 'Falha ao enviar para a API do parceiro.',
                    'status' => $result['status'],
                ], 502);
            }
            return response()->json([
                'message' => 'Proposta enviada para a API do parceiro.',
                'status' => $result['status'],
            ]);
        }

        if (!$parceiro->contato_email) {
            return response()->json(['message' => 'Email do parceiro nao configurado.'], 422);
        }

        $this->sendParceiroEmail($parceiro, $data);

        return response()->json(['message' => 'Proposta enviada por email ao parceiro.']);
    }

    private function loadExternalSuggestion(Cliente $cliente, ?Lead $lead): ?object
    {
        $table = self::EXTERNAL_DB.'.simulacoes_imobiliarias';

        if ($lead && $lead->external_id) {
            return DB::table($table)
                ->select(['id', 'valor_emprestimo', 'prazo'])
                ->where('id', $lead->external_id)
                ->first();
        }

        $document = $this->normalizeDocument($cliente->cpf_cnpj ?? '');
        if ($document === '') {
            return null;
        }

        return DB::table($table.' as s')
            ->select(['s.id', 's.valor_emprestimo', 's.prazo'])
            ->whereRaw(
                "REPLACE(REPLACE(REPLACE(REPLACE(s.cpf, '.', ''), '-', ''), '/', ''), ' ', '') = ?",
                [$document]
            )
            ->orderByDesc('s.id')
            ->first();
    }

    private function normalizeDocument(string $value): string
    {
        return preg_replace('/\D+/', '', $value);
    }

    private function buildParceiroPayload(PropostaCredito $proposta): array
    {
        $cliente = $proposta->cliente;
        $documentos = $this->mapDocumentos(
            PropostaDocumento::query()->where('proposta_id', $proposta->id)->orderBy('id')->get()
        );

        return [
            'proposta' => [
                'id' => $proposta->id,
                'numero' => $this->formatPropostaNumero($proposta->id),
                'valor_solicitado' => $proposta->valor_solicitado,
                'prazo_meses' => $proposta->prazo_meses,
                'finalidade' => $proposta->finalidade,
                'status' => $proposta->status ?? null,
            ],
            'cliente' => [
                'id' => $cliente?->id,
                'nome' => $cliente?->nome_razao ?? $cliente?->nome,
                'documento' => $cliente?->cpf_cnpj,
                'tipo' => $cliente?->tipo_cliente,
            ],
            'documentos' => $documentos,
        ];
    }

    private function mapDocumentos($documentos): array
    {
        $items = [];
        foreach ($documentos as $documento) {
            $items[] = [
                'id' => $documento->id,
                'tipo' => $documento->tipo,
                'status' => $documento->status,
                'arquivo' => $documento->arquivo,
                'url' => $this->documentUrl($documento->arquivo),
                'criado_em' => $documento->created_at,
            ];
        }
        return $items;
    }

    private function mapSolicitacoes($solicitacoes): array
    {
        $items = [];
        foreach ($solicitacoes as $solicitacao) {
            $items[] = [
                'id' => $solicitacao->id,
                'token' => $solicitacao->token,
                'canal' => $solicitacao->canal,
                'destinatario' => $solicitacao->destinatario,
                'status' => $solicitacao->status,
                'tipo_cliente' => $solicitacao->tipo_cliente,
                'created_at' => $solicitacao->created_at,
            ];
        }
        return $items;
    }

    private function documentUrl(?string $path): ?string
    {
        if (!$path) {
            return null;
        }
        $path = ltrim($path, '/');
        $path = preg_replace('#^public/#', '', $path);
        $base = rtrim(config('app.url') ?: request()->getSchemeAndHttpHost(), '/');

        return $base.'/public/'.$path;
    }

    private function formatPropostaNumero(?int $id): string
    {
        if (!$id) {
            return 'PR-0000';
        }
        return 'PR-'.str_pad((string) $id, 4, '0', STR_PAD_LEFT);
    }

    private function parseApiHeaders(?string $headers): array
    {
        if (!$headers) {
            return [];
        }
        $decoded = json_decode($headers, true);
        if (!is_array($decoded)) {
            return [];
        }
        return $decoded;
    }

    private function sendParceiroApi(Parceiro $parceiro, array $data): array
    {
        $headers = $this->parseApiHeaders($parceiro->api_headers);
        if ($parceiro->api_key) {
            $headers['X-API-KEY'] = $parceiro->api_key;
        }

        $request = Http::timeout(20)->withHeaders($headers);
        if ($parceiro->api_token) {
            $request = $request->withToken($parceiro->api_token);
        }

        $response = $request->post($parceiro->api_url, $data);
        if (!$response->successful()) {
            return [
                'ok' => false,
                'status' => $response->status(),
                'body' => $response->body(),
            ];
        }

        return [
            'ok' => true,
            'status' => $response->status(),
            'body' => $response->json() ?? $response->body(),
        ];
    }

    private function sendParceiroEmail(Parceiro $parceiro, array $data): void
    {
        $propostaNumero = $data['proposta']['numero'] ?? $this->formatPropostaNumero($data['proposta']['id'] ?? null);
        $clienteNome = $data['cliente']['nome'] ?? 'Cliente';
        $subject = 'LPC Capital - Proposta '.$propostaNumero.' - '.$clienteNome;

        $html = $this->buildParceiroEmailHtml($parceiro, $data);
        $text = $this->buildParceiroEmailText($parceiro, $data);

        $fromAddress = config('mail.from.address') ?: env('MAIL_FROM_ADDRESS');
        $fromName = config('mail.from.name') ?: env('MAIL_FROM_NAME');

        Mail::send([], [], function ($message) use ($parceiro, $subject, $fromAddress, $fromName, $html, $text) {
            if ($fromAddress) {
                $message->from($fromAddress, $fromName ?: null);
            }
            $message->to($parceiro->contato_email)
                ->replyTo('contato@marianalpccapital.com.br', 'LPC Capital')
                ->subject($subject)
                ->html($html)
                ->text($text);
        });
    }

    private function buildParceiroEmailHtml(Parceiro $parceiro, array $data): string
    {
        $cliente = $data['cliente'] ?? [];
        $proposta = $data['proposta'] ?? [];
        $documentos = $data['documentos'] ?? [];
        $lista = '';
        foreach ($documentos as $doc) {
            $url = $doc['url'] ?? '';
            $label = $doc['tipo'] ?? '-';
            $linha = $url ? '<a href="'.e($url).'">'.e($label).'</a>' : e($label);
            $lista .= '<li>'.$linha.'</li>';
        }

        $html = '<!doctype html><html><head><meta charset="utf-8">';
        $html .= '<style>body{font-family:Arial,sans-serif;color:#0f172a;margin:24px}';
        $html .= 'h1{font-size:22px;margin:0}h2{font-size:13px;margin:18px 0 8px;text-transform:uppercase;letter-spacing:.12em;color:#475569}';
        $html .= 'ul{padding-left:18px}li{margin-bottom:6px;font-size:13px}.card{border:1px solid #e2e8f0;border-radius:12px;padding:16px;margin-top:16px}';
        $html .= 'a{color:#0ea5e9;text-decoration:none}</style></head><body>';
        $html .= '<h1>Proposta '.$this->formatEmailValue($proposta['numero'] ?? '').'</h1>';
        $html .= '<p>Parceiro: '.$this->formatEmailValue($parceiro->nome).'</p>';
        $html .= '<div class="card"><h2>Dados do cliente</h2>';
        $html .= '<p><strong>Nome:</strong> '.$this->formatEmailValue($cliente['nome'] ?? '').'</p>';
        $html .= '<p><strong>Documento:</strong> '.$this->formatEmailValue($cliente['documento'] ?? '').'</p>';
        $html .= '<p><strong>Tipo:</strong> '.$this->formatEmailValue($cliente['tipo'] ?? '').'</p></div>';
        $html .= '<div class="card"><h2>Resumo da proposta</h2>';
        $html .= '<p><strong>Valor solicitado:</strong> '.$this->formatEmailValue($this->formatCurrency($proposta['valor_solicitado'] ?? null)).'</p>';
        $html .= '<p><strong>Prazo:</strong> '.$this->formatEmailValue($proposta['prazo_meses'] ?? null).'</p>';
        $html .= '<p><strong>Finalidade:</strong> '.$this->formatEmailValue($proposta['finalidade'] ?? '').'</p></div>';
        $html .= '<div class="card"><h2>Documentos enviados</h2><ul>'.$lista.'</ul></div>';
        $html .= '<p>Equipe LPC Capital</p>';
        $html .= '</body></html>';

        return $html;
    }

    private function buildParceiroEmailText(Parceiro $parceiro, array $data): string
    {
        $cliente = $data['cliente'] ?? [];
        $proposta = $data['proposta'] ?? [];
        $documentos = $data['documentos'] ?? [];

        $lines = [
            'Proposta '.$this->formatEmailValue($proposta['numero'] ?? ''),
            'Parceiro: '.$this->formatEmailValue($parceiro->nome),
            '',
            'Dados do cliente:',
            'Nome: '.$this->formatEmailValue($cliente['nome'] ?? ''),
            'Documento: '.$this->formatEmailValue($cliente['documento'] ?? ''),
            'Tipo: '.$this->formatEmailValue($cliente['tipo'] ?? ''),
            '',
            'Resumo da proposta:',
            'Valor solicitado: '.$this->formatEmailValue($this->formatCurrency($proposta['valor_solicitado'] ?? null)),
            'Prazo (meses): '.$this->formatEmailValue($proposta['prazo_meses'] ?? null),
            'Finalidade: '.$this->formatEmailValue($proposta['finalidade'] ?? ''),
            '',
            'Documentos enviados:',
        ];

        foreach ($documentos as $doc) {
            $label = $doc['tipo'] ?? '-';
            $url = $doc['url'] ?? '';
            $lines[] = $url ? '- '.$label.' ('.$url.')' : '- '.$label;
        }

        $lines[] = '';
        $lines[] = 'Equipe LPC Capital';

        return implode("\n", $lines);
    }

    private function formatEmailValue($value): string
    {
        if ($value === null) {
            return '-';
        }
        $text = trim((string) $value);
        return $text === '' ? '-' : $text;
    }

    private function formatCurrency($value): string
    {
        if ($value === null || $value === '') {
            return '-';
        }
        $number = is_numeric($value) ? (float) $value : null;
        if ($number === null) {
            return (string) $value;
        }
        return 'R$ '.number_format($number, 2, ',', '.');
    }

    private function resolveDefaultStatusId(): ?int
    {
        try {
            if (!Schema::hasTable('status_proposta')) {
                return null;
            }
        } catch (\Throwable $exception) {
            return null;
        }

        $preferred = DB::table('status_proposta')
            ->select(['id', 'nome'])
            ->orderBy('id')
            ->get();

        if ($preferred->isEmpty()) {
            return null;
        }

        foreach ($preferred as $row) {
            $name = strtolower(trim($row->nome ?? ''));
            if (in_array($name, ['rascunho', 'novo', 'nova'], true)) {
                return (int) $row->id;
            }
        }

        return isset($preferred[0]) ? (int) $preferred[0]->id : null;
    }
}
