<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class SimulacaoFormController extends Controller
{
    private function getFormConfig(string $form): ?array
    {
        $configs = [
            'imobiliarias' => [
                'table' => 'simulacoes_imobiliarias',
                'fields' => [
                    'nome', 'email', 'telefone', 'cpf', 'renda', 'cep', 'endereco', 'numero',
                    'bairro', 'complemento', 'cidade', 'estado', 'valor_imovel', 'tipo_imovel',
                    'propriedade', 'metragem', 'financiado', 'saldo_financiado', 'condominio',
                    'valor_emprestimo', 'prazo', 'aceite', 'origem', 'ip', 'user_agent',
                ],
                'required' => [
                    'nome', 'email', 'telefone', 'cpf', 'renda', 'cep', 'endereco', 'numero',
                    'bairro', 'cidade', 'estado', 'valor_imovel', 'tipo_imovel', 'propriedade',
                    'metragem', 'financiado', 'condominio', 'valor_emprestimo', 'prazo', 'aceite',
                ],
                'digits' => ['cpf', 'cep', 'telefone'],
                'numeric' => ['renda', 'valor_imovel', 'saldo_financiado', 'valor_emprestimo'],
                'int' => ['metragem', 'prazo'],
                'bool' => ['aceite'],
            ],
            'auto_equity_pf' => [
                'table' => 'simulacoes_auto_equity_pf',
                'fields' => [
                    'nome', 'data_nascimento', 'cpf', 'email', 'telefone', 'nome_mae', 'rg_cnh',
                    'orgao_emissor', 'uf_emissor', 'naturalidade', 'nacionalidade', 'profissao',
                    'tempo_atividade', 'renda', 'cep', 'endereco', 'numero', 'bairro', 'complemento',
                    'cidade', 'estado', 'valor_emprestimo', 'veiculo_tipo', 'veiculo_marca',
                    'veiculo_modelo', 'veiculo_versao', 'ano_fabricacao', 'ano_modelo', 'placa',
                    'veiculo_nome', 'alienacao', 'saldo_devedor', 'instituicao_saldo', 'origem',
                    'ip', 'user_agent',
                ],
                'required' => [
                    'nome', 'data_nascimento', 'cpf', 'email', 'telefone', 'nome_mae', 'rg_cnh',
                    'orgao_emissor', 'uf_emissor', 'naturalidade', 'nacionalidade', 'profissao',
                    'tempo_atividade', 'renda', 'cep', 'endereco', 'numero', 'bairro', 'cidade',
                    'estado', 'valor_emprestimo', 'veiculo_tipo', 'veiculo_marca', 'veiculo_modelo',
                    'ano_fabricacao', 'ano_modelo', 'placa', 'veiculo_nome', 'alienacao',
                ],
                'digits' => ['cpf', 'cep', 'telefone', 'placa'],
                'numeric' => ['renda', 'valor_emprestimo', 'saldo_devedor'],
                'int' => ['ano_fabricacao', 'ano_modelo'],
                'date' => ['data_nascimento'],
            ],
            'auto_equity_pj' => [
                'table' => 'simulacoes_auto_equity_pj',
                'fields' => [
                    'nome', 'cpf', 'empresa_nome', 'empresa_cnpj', 'email', 'telefone', 'faturamento_12m',
                    'valor_emprestimo', 'cep', 'endereco', 'numero', 'bairro', 'complemento', 'cidade',
                    'estado', 'socio_nome', 'socio_cpf', 'socio_data_nascimento', 'socio_nome_mae',
                    'socio_rg_cnh', 'socio_endereco', 'socio_naturalidade', 'socio_nacionalidade',
                    'socio_renda', 'veiculo_marca', 'veiculo_modelo', 'veiculo_versao', 'ano_fabricacao',
                    'ano_modelo', 'placa', 'veiculo_nome_empresa', 'origem', 'ip', 'user_agent',
                ],
                'required' => [
                    'nome', 'cpf', 'empresa_nome', 'empresa_cnpj', 'email', 'telefone', 'faturamento_12m',
                    'valor_emprestimo', 'cep', 'endereco', 'numero', 'bairro', 'cidade', 'estado',
                    'socio_nome', 'socio_cpf', 'socio_data_nascimento', 'socio_nome_mae', 'socio_rg_cnh',
                    'socio_endereco', 'socio_naturalidade', 'socio_nacionalidade', 'socio_renda',
                    'veiculo_marca', 'veiculo_modelo', 'ano_fabricacao', 'ano_modelo', 'placa',
                    'veiculo_nome_empresa',
                ],
                'digits' => ['cpf', 'empresa_cnpj', 'socio_cpf', 'cep', 'telefone', 'placa'],
                'numeric' => ['faturamento_12m', 'valor_emprestimo', 'socio_renda'],
                'int' => ['ano_fabricacao', 'ano_modelo'],
                'date' => ['socio_data_nascimento'],
            ],
            'consignado' => [
                'table' => 'simulacoes_consignado',
                'fields' => [
                    'nome', 'data_nascimento', 'cpf', 'email', 'telefone', 'valor_emprestimo',
                    'renda_mensal', 'situacao_profissional', 'motivo_emprestimo', 'cep', 'endereco',
                    'numero', 'bairro', 'complemento', 'cidade', 'estado', 'veiculo_nome_cliente',
                    'placa', 'renavam', 'ano_fabricacao', 'ano_modelo', 'marca', 'modelo',
                    'pacote_acessorios', 'versao', 'autorizacao_scr', 'origem', 'ip', 'user_agent',
                ],
                'required' => [
                    'nome', 'data_nascimento', 'cpf', 'email', 'telefone', 'valor_emprestimo',
                    'renda_mensal', 'situacao_profissional', 'motivo_emprestimo', 'cep', 'endereco',
                    'numero', 'bairro', 'cidade', 'estado', 'veiculo_nome_cliente', 'placa', 'renavam',
                    'ano_fabricacao', 'ano_modelo', 'marca', 'modelo', 'autorizacao_scr',
                ],
                'digits' => ['cpf', 'cep', 'telefone', 'renavam', 'placa'],
                'numeric' => ['valor_emprestimo', 'renda_mensal'],
                'int' => ['ano_fabricacao', 'ano_modelo'],
                'bool' => ['autorizacao_scr'],
                'date' => ['data_nascimento'],
            ],
            'consorcio' => [
                'table' => 'simulacoes_consorcio',
                'fields' => [
                    'tipo_pessoa', 'nome', 'cpf', 'email', 'telefone', 'renda', 'tipo_bem', 'valor_bem',
                    'valor_emprestimo', 'lance_embutido', 'lance_proprio', 'valor_parcela_max',
                    'prazo_desejado', 'origem', 'ip', 'user_agent',
                ],
                'required' => [
                    'tipo_pessoa', 'nome', 'cpf', 'email', 'telefone', 'renda', 'tipo_bem', 'valor_bem',
                    'valor_emprestimo', 'lance_embutido', 'lance_proprio', 'valor_parcela_max',
                    'prazo_desejado',
                ],
                'digits' => ['cpf', 'telefone'],
                'numeric' => ['renda', 'valor_bem', 'valor_emprestimo', 'valor_parcela_max'],
            ],
            'credito_aval_digital_fgi' => [
                'table' => 'simulacoes_credito_aval_digital_fgi',
                'fields' => [
                    'nome', 'cpf', 'razao_social', 'cnpj', 'email', 'telefone', 'faturamento_anual',
                    'valor_solicitado', 'valor_emprestimo', 'prazo_meses', 'segmento', 'segmento_outros',
                    'funcionarios', 'observacoes', 'origem', 'ip', 'user_agent',
                ],
                'required' => [
                    'nome', 'cpf', 'razao_social', 'cnpj', 'email', 'telefone', 'faturamento_anual',
                    'valor_solicitado', 'valor_emprestimo', 'prazo_meses', 'segmento', 'funcionarios',
                ],
                'digits' => ['cpf', 'cnpj', 'telefone'],
                'numeric' => ['faturamento_anual', 'valor_solicitado', 'valor_emprestimo'],
                'int' => ['prazo_meses', 'funcionarios'],
            ],
            'credito_condominio' => [
                'table' => 'simulacoes_credito_condominio',
                'fields' => [
                    'nome', 'cpf', 'condominio_nome', 'condominio_cnpj', 'email', 'telefone', 'site',
                    'contato', 'cep', 'endereco', 'cidade', 'estado', 'valor_imovel', 'qtd_torres',
                    'total_unidades', 'metragem_media', 'inadimplencia_12m', 'arrecadacao_mensal',
                    'forma_custeio', 'gastos_energia', 'ano_inicio', 'admin_nome', 'admin_cnpj',
                    'admin_telefone', 'admin_email', 'admin_site', 'admin_contato', 'sindico_nome',
                    'sindico_profissional', 'sindico_cpf', 'sindico_email', 'sindico_celular',
                    'valor_credito', 'valor_emprestimo', 'carencia', 'finalidade',
                    'forma_custeio_credito', 'economia_mensal', 'origem', 'ip', 'user_agent',
                ],
                'required' => [
                    'nome', 'cpf', 'condominio_nome', 'condominio_cnpj', 'email', 'telefone', 'cep',
                    'endereco', 'cidade', 'estado', 'valor_imovel', 'qtd_torres', 'total_unidades',
                    'metragem_media', 'inadimplencia_12m', 'arrecadacao_mensal', 'forma_custeio',
                    'admin_nome', 'admin_cnpj', 'admin_telefone', 'admin_email', 'sindico_nome',
                    'sindico_profissional', 'sindico_cpf', 'sindico_email', 'sindico_celular',
                    'valor_credito', 'valor_emprestimo', 'carencia', 'finalidade',
                    'forma_custeio_credito',
                ],
                'digits' => ['cpf', 'condominio_cnpj', 'admin_cnpj', 'telefone', 'cep', 'sindico_cpf'],
                'numeric' => [
                    'valor_imovel', 'metragem_media', 'inadimplencia_12m', 'arrecadacao_mensal',
                    'gastos_energia', 'valor_credito', 'valor_emprestimo', 'economia_mensal',
                ],
                'int' => ['qtd_torres', 'total_unidades'],
            ],
            'credito_construcao' => [
                'table' => 'simulacoes_credito_construcao',
                'fields' => [
                    'tipo_pessoa', 'nome', 'cpf', 'email', 'telefone', 'valor_proposta', 'valor_emprestimo',
                    'valor_imovel_acabado', 'tipo_construcao', 'matricula_iptu', 'percentual_construcao',
                    'alvara', 'art_rrt', 'projeto_arquitetonico', 'cep', 'endereco', 'numero', 'bairro',
                    'complemento', 'cidade', 'estado', 'origem', 'ip', 'user_agent',
                ],
                'required' => [
                    'tipo_pessoa', 'nome', 'cpf', 'email', 'telefone', 'valor_proposta', 'valor_emprestimo',
                    'valor_imovel_acabado', 'tipo_construcao', 'matricula_iptu', 'percentual_construcao',
                    'alvara', 'art_rrt', 'projeto_arquitetonico', 'cep', 'endereco', 'numero', 'bairro',
                    'cidade', 'estado',
                ],
                'digits' => ['cpf', 'telefone', 'cep'],
                'numeric' => ['valor_proposta', 'valor_emprestimo', 'valor_imovel_acabado'],
            ],
            'creditos_pj_diversos' => [
                'table' => 'simulacoes_creditos_pj_diversos',
                'fields' => [
                    'nome', 'cpf', 'razao_social', 'cnpj', 'email', 'telefone', 'tipo_operacao',
                    'faturamento_anual', 'valor_proposta', 'valor_emprestimo', 'valor_bem_arrendado',
                    'tipo_garantia', 'socio_nome', 'socio_cpf', 'socio_email', 'bancos_relacionamento',
                    'parecer_credito', 'observacoes', 'origem', 'ip', 'user_agent',
                ],
                'required' => [
                    'nome', 'cpf', 'razao_social', 'cnpj', 'email', 'telefone', 'tipo_operacao',
                    'faturamento_anual', 'valor_proposta', 'valor_emprestimo',
                ],
                'digits' => ['cpf', 'cnpj', 'telefone', 'socio_cpf'],
                'numeric' => ['faturamento_anual', 'valor_proposta', 'valor_emprestimo', 'valor_bem_arrendado'],
            ],
            'financiamento_imobiliario_pf' => [
                'table' => 'simulacoes_financiamento_imobiliario_pf',
                'fields' => [
                    'nome', 'cpf', 'email', 'telefone', 'data_nascimento', 'renda', 'somar_renda',
                    'tipo_imovel', 'uf_imovel', 'valor_imovel', 'valor_entrada', 'valor_financiamento',
                    'valor_emprestimo', 'prazo', 'imovel_definido', 'usar_fgts', 'incluir_itbi',
                    'valor_despesas', 'valor_total', 'origem', 'ip', 'user_agent',
                ],
                'required' => [
                    'nome', 'cpf', 'email', 'telefone', 'data_nascimento', 'renda', 'somar_renda',
                    'tipo_imovel', 'uf_imovel', 'valor_imovel', 'valor_financiamento', 'valor_emprestimo',
                    'prazo', 'imovel_definido', 'usar_fgts', 'incluir_itbi',
                ],
                'digits' => ['cpf', 'telefone'],
                'numeric' => ['renda', 'valor_imovel', 'valor_entrada', 'valor_financiamento', 'valor_emprestimo', 'valor_despesas', 'valor_total'],
                'int' => ['prazo'],
                'date' => ['data_nascimento'],
            ],
            'financiamento_veiculo' => [
                'table' => 'simulacoes_financiamento_veiculo',
                'fields' => [
                    'tipo_pessoa', 'nome', 'cpf', 'email', 'telefone', 'data_nascimento', 'genero',
                    'estado_civil', 'nome_mae', 'renda', 'cnpj', 'faturamento_anual', 'representante_nome',
                    'representante_cpf', 'representante_email', 'representante_whatsapp', 'representante_nascimento',
                    'representante_telefone', 'condicao_veiculo', 'possui_habilitacao', 'uso_comercial',
                    'tipo_veiculo', 'marca', 'modelo', 'ano_fabricacao', 'ano_modelo', 'combustivel',
                    'uf_licenciamento', 'valor_veiculo', 'valor_entrada', 'valor_emprestimo',
                    'origem', 'ip', 'user_agent',
                ],
                'required' => [
                    'tipo_pessoa', 'nome', 'email', 'telefone', 'condicao_veiculo', 'possui_habilitacao',
                    'uso_comercial', 'tipo_veiculo', 'marca', 'modelo', 'ano_fabricacao', 'ano_modelo',
                    'combustivel', 'uf_licenciamento', 'valor_veiculo', 'valor_emprestimo',
                ],
                'digits' => ['cpf', 'cnpj', 'telefone', 'representante_cpf', 'representante_whatsapp', 'representante_telefone'],
                'numeric' => ['renda', 'faturamento_anual', 'valor_veiculo', 'valor_entrada', 'valor_emprestimo'],
                'int' => ['ano_fabricacao', 'ano_modelo'],
                'date' => ['data_nascimento', 'representante_nascimento'],
            ],
        ];

        return $configs[$form] ?? null;
    }

    private function normalizeDigits($value): ?string
    {
        if ($value === null) {
            return null;
        }
        return preg_replace('/\D+/', '', (string) $value);
    }

    private function normalizeNumber($value): ?float
    {
        if ($value === null || $value === '') {
            return null;
        }
        if (is_numeric($value)) {
            return (float) $value;
        }
        $clean = preg_replace('/[^0-9,\.\-]/', '', (string) $value);
        if (strpos($clean, ',') !== false && strpos($clean, '.') === false) {
            $clean = str_replace(',', '.', $clean);
        } else {
            $clean = str_replace(',', '', $clean);
        }
        return is_numeric($clean) ? (float) $clean : null;
    }

    private function normalizeText($value): ?string
    {
        if ($value === null) {
            return null;
        }
        $text = trim((string) $value);
        return $text === '' ? null : $text;
    }

    public function upload(Request $request)
    {
        $files = $request->file('documentos', []);
        if (!$files) {
            return response()->json(['success' => false, 'message' => 'Nenhum arquivo enviado.'], 422);
        }
        if (!is_array($files)) {
            $files = [$files];
        }

        $uploadDir = '/home/ipccapitalmarian/public_html/formulario/api/uploads';
        if (!is_dir($uploadDir)) {
            @mkdir($uploadDir, 0755, true);
        }

        $saved = [];
        foreach ($files as $file) {
            if (!$file || !$file->isValid()) {
                continue;
            }
            if ($file->getSize() > 5 * 1024 * 1024) {
                continue;
            }
            $ext = strtolower($file->getClientOriginalExtension());
            if (!in_array($ext, ['pdf', 'jpg', 'jpeg', 'png'], true)) {
                continue;
            }
            $filename = Str::random(16) . '_' . time() . '.' . $ext;
            $file->move($uploadDir, $filename);
            $saved[] = [
                'arquivo' => $filename,
                'url' => '/formulario/api/uploads/' . $filename,
            ];
        }

        if (!$saved) {
            return response()->json(['success' => false, 'message' => 'Nenhum arquivo válido enviado.'], 422);
        }

        return response()->json(['success' => true, 'files' => $saved]);
    }

    public function store(string $form, Request $request)
    {
        $config = $this->getFormConfig($form);
        if (!$config) {
            return response()->json(['success' => false, 'message' => 'Formulário inválido.'], 404);
        }

        $rules = [];
        foreach ($config['fields'] as $field) {
            $isRequired = in_array($field, $config['required'], true);
            $rule = $isRequired ? 'required' : 'nullable';
            if (!empty($config['numeric']) && in_array($field, $config['numeric'], true)) {
                $rule .= '|numeric';
            }
            if (!empty($config['int']) && in_array($field, $config['int'], true)) {
                $rule .= '|integer';
            }
            if (!empty($config['bool']) && in_array($field, $config['bool'], true)) {
                $rule .= '|boolean';
            }
            if (!empty($config['date']) && in_array($field, $config['date'], true)) {
                $rule .= '|date';
            }
            $rules[$field] = $rule;
        }

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => $validator->errors()->first()], 422);
        }

        $payload = [];
        foreach ($config['fields'] as $field) {
            if (in_array($field, ['origem', 'ip', 'user_agent'], true)) {
                continue;
            }
            $value = $request->input($field);
            if (!empty($config['digits']) && in_array($field, $config['digits'], true)) {
                $value = $this->normalizeDigits($value);
            }
            if (!empty($config['numeric']) && in_array($field, $config['numeric'], true)) {
                $value = $this->normalizeNumber($value);
            } elseif (!empty($config['int']) && in_array($field, $config['int'], true)) {
                $value = $value === null || $value === '' ? null : (int) $value;
            } elseif (!empty($config['bool']) && in_array($field, $config['bool'], true)) {
                $value = (bool) $value;
            } else {
                $value = $this->normalizeText($value);
            }
            if ($value !== null || in_array($field, $config['required'], true)) {
                $payload[$field] = $value;
            }
        }

        $payload['origem'] = $request->header('origin') ?: $request->header('referer');
        $payload['ip'] = $request->ip();
        $payload['user_agent'] = (string) $request->userAgent();

        $id = DB::connection('external_mysql')->table($config['table'])->insertGetId($payload);

        $docs = $request->input('documentos', []);
        if (is_array($docs) && $docs) {
            foreach ($docs as $doc) {
                $arquivo = is_array($doc) ? ($doc['arquivo'] ?? null) : $doc;
                if (!$arquivo) {
                    continue;
                }
                $tipo = is_array($doc) ? ($doc['tipo'] ?? null) : null;
                DB::connection('external_mysql')->table('simulacoes_documentos')->insert([
                    'simulacao_id' => $id,
                    'arquivo' => $arquivo,
                    'tipo' => $tipo,
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Formulário salvo com sucesso!',
            'id' => $id,
        ]);
    }
}
