<?php

namespace App\Services;

use App\Models\Configuracao;
use Illuminate\Support\Facades\Http;

class SmsService
{
    private const DEFAULT_API_URL = 'https://sms.comtele.com.br/api/v2/send';

    public function normalizePhone(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }
        $digits = preg_replace('/\D+/', '', $value);
        if (!$digits || strlen($digits) < 10) {
            return null;
        }
        return $digits;
    }

    public function config(): array
    {
        $config = Configuracao::query()->first();

        return [
            'api_url' => $config?->sms_api_url ?: env('COMTELE_SMS_API_URL', self::DEFAULT_API_URL),
            'auth_key' => $config?->sms_auth_key ?: env('COMTELE_SMS_AUTH_KEY'),
            'sender' => $config?->sms_sender ?: env('COMTELE_SMS_SENDER'),
        ];
    }

    public function send(string $phone, string $message, ?string $sender = null): array
    {
        $normalized = $this->normalizePhone($phone);
        if (!$normalized) {
            return [
                'sent' => false,
                'error' => 'Telefone invalido.',
            ];
        }

        $config = $this->config();
        $apiUrl = $config['api_url'];
        $authKey = $config['auth_key'];
        if (!$apiUrl || !$authKey) {
            return [
                'sent' => false,
                'error' => 'Credenciais de SMS nao configuradas.',
            ];
        }

        $payload = [
            'Receivers' => $normalized,
            'Content' => $message,
        ];
        $sender = $sender ?: $config['sender'];
        if ($sender) {
            $payload['Sender'] = $sender;
        }

        try {
            $response = Http::timeout(12)
                ->withHeaders([
                    'auth-key' => $authKey,
                    'Content-Type' => 'application/json',
                ])
                ->post($apiUrl, $payload);
        } catch (\Throwable $exception) {
            return [
                'sent' => false,
                'error' => $exception->getMessage(),
            ];
        }

        if (!$response->successful()) {
            return [
                'sent' => false,
                'error' => 'Falha ao enviar SMS.',
            ];
        }

        $body = $response->json();
        if (is_array($body) && array_key_exists('Success', $body) && !$body['Success']) {
            return [
                'sent' => false,
                'error' => 'Erro retornado pela API.',
            ];
        }

        return [
            'sent' => true,
        ];
    }
}
