from __future__ import annotations

import tempfile
from pathlib import Path
from typing import Any, Dict

from fastapi import FastAPI, File, UploadFile
from pydantic import BaseModel


app = FastAPI(title="Docling Service", version="0.1.0")
converter = None
docling_available = None
DocumentConverter = None


def _get_converter():
    global converter, docling_available, DocumentConverter
    if DocumentConverter is None:
        try:
            from docling.document_converter import DocumentConverter as _DocumentConverter
        except Exception:
            docling_available = False
            return None
        DocumentConverter = _DocumentConverter
        docling_available = True
    if converter is None and docling_available:
        converter = DocumentConverter()
    return converter


class PathRequest(BaseModel):
    path: str


def _doc_to_payload(doc) -> Dict[str, Any]:
    markdown = doc.export_to_markdown()
    text = markdown
    if hasattr(doc, "export_to_text"):
        try:
            text = doc.export_to_text()
        except Exception:
            text = markdown
    return {
        "markdown": markdown,
        "text": text,
    }


def _convert(path: str) -> Dict[str, Any]:
    local_converter = _get_converter()
    if local_converter is None:
        return {"success": False, "error": "Docling nao instalado no ambiente."}

    try:
        result = local_converter.convert(path)
    except Exception as exc:  # pragma: no cover - runtime errors
        return {"success": False, "error": str(exc)}

    doc = getattr(result, "document", None)
    if doc is None:
        return {"success": False, "error": "Documento vazio."}

    payload = _doc_to_payload(doc)
    payload["success"] = True
    payload["meta"] = {
        "input_path": path,
    }
    return payload


@app.get("/health")
def health() -> Dict[str, Any]:
    return {
        "ok": True,
        "docling_ready": docling_available,
        "docling_initialized": converter is not None,
    }


@app.post("/extract-path")
def extract_path(payload: PathRequest) -> Dict[str, Any]:
    return _convert(payload.path)


@app.post("/extract")
async def extract_file(file: UploadFile = File(...)) -> Dict[str, Any]:
    suffix = Path(file.filename or "documento").suffix
    with tempfile.NamedTemporaryFile(delete=False, suffix=suffix) as tmp:
        tmp.write(await file.read())
        tmp_path = tmp.name

    return _convert(tmp_path)
