<?php

namespace App\Http\Controllers;

use App\Models\AgendaEvento;
use App\Models\AgendaEventoParticipante;
use App\Services\AgendaService;
use Carbon\Carbon;
use Illuminate\Http\Request;

class AgendaParticipanteController extends Controller
{
    public function __construct(private AgendaService $service)
    {
    }

    public function store(Request $request, AgendaEvento $agenda)
    {
        $user = $request->user();
        if (!$this->service->canEdit($user, $agenda)) {
            return response()->json(['message' => 'Sem permissao para editar.'], 403);
        }

        $data = $request->validate([
            'participantes' => ['required', 'array'],
            'participantes.*' => ['integer', 'exists:users,id'],
        ]);

        $ids = array_values(array_unique(array_map('intval', $data['participantes'])));
        foreach ($ids as $id) {
            if ($id === (int) $agenda->owner_id) {
                continue;
            }
            AgendaEventoParticipante::query()->firstOrCreate([
                'evento_id' => $agenda->id,
                'usuario_id' => $id,
            ], [
                'status' => 'pendente',
            ]);
        }

        return response()->json($agenda->load('participantes.usuario:id,name,email'));
    }

    public function update(Request $request, AgendaEvento $agenda, AgendaEventoParticipante $participante)
    {
        if ((int) $participante->evento_id !== (int) $agenda->id) {
            return response()->json(['message' => 'Participante invalido.'], 422);
        }

        $user = $request->user();
        $canEdit = $this->service->canEdit($user, $agenda);
        $isSelf = (int) $participante->usuario_id === (int) $user->id;
        if (!$canEdit && !$isSelf) {
            return response()->json(['message' => 'Sem permissao para editar.'], 403);
        }

        $data = $request->validate([
            'status' => ['required', 'string', 'in:pendente,aceito,recusado'],
        ]);

        $participante->status = $data['status'];
        $participante->respondido_em = Carbon::now();
        $participante->save();

        return response()->json($agenda->load('participantes.usuario:id,name,email'));
    }

    public function destroy(Request $request, AgendaEvento $agenda, AgendaEventoParticipante $participante)
    {
        if ((int) $participante->evento_id !== (int) $agenda->id) {
            return response()->json(['message' => 'Participante invalido.'], 422);
        }

        $user = $request->user();
        $canEdit = $this->service->canEdit($user, $agenda);
        $isSelf = (int) $participante->usuario_id === (int) $user->id;
        if (!$canEdit && !$isSelf) {
            return response()->json(['message' => 'Sem permissao para remover.'], 403);
        }

        $participante->delete();

        return response()->json($agenda->load('participantes.usuario:id,name,email'));
    }
}
