<?php

namespace App\Http\Controllers;

use App\Models\Cliente;
use App\Models\PessoaFisica;
use App\Models\PessoaJuridica;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PessoaJuridicaController extends Controller
{
    private const ESTADO_CIVIL_CASADO = ['CASADO', 'UNIAO_ESTAVEL'];

    public function index()
    {
        $pessoas = PessoaJuridica::query()
            ->with('cliente')
            ->orderByDesc('cliente_id')
            ->paginate(20);

        return response()->json($pessoas);
    }

    public function create()
    {
        return response()->json(['message' => 'create']);
    }

    public function store(Request $request)
    {
        $clienteId = $this->resolveCliente($request, 'PJ');
        $estadoCivilSocio = $this->normalizeEstadoCivil($request->input('estado_civil_socio'));

        $registro = PessoaJuridica::query()->create(array_merge(
            $request->only([
                'cnpj',
                'data_abertura',
                'ramo_atuacao',
                'faturamento_anual',
                'renda',
                'estado_civil_socio',
                'conjuge_nome',
            ]),
            [
                'cliente_id' => $clienteId,
                'estado_civil_socio' => $estadoCivilSocio ?: $request->input('estado_civil_socio'),
            ]
        ));

        $this->storeEndereco($clienteId, $request);
        $this->ensureConjuge($estadoCivilSocio, $request);

        return response()->json($registro->load('cliente'), 201);
    }

    public function show(int $id)
    {
        $pessoa = PessoaJuridica::query()
            ->with('cliente')
            ->findOrFail($id);

        return response()->json($pessoa);
    }

    public function edit(int $id)
    {
        return response()->json(PessoaJuridica::query()->findOrFail($id));
    }

    public function update(Request $request, int $id)
    {
        $registro = PessoaJuridica::query()->findOrFail($id);
        $registro->update($request->only([
            'cnpj',
            'data_abertura',
            'ramo_atuacao',
            'faturamento_anual',
            'renda',
            'estado_civil_socio',
            'conjuge_nome',
        ]));

        $clienteData = $request->only([
            'nome_razao',
            'cpf_cnpj',
            'data_nascimento',
            'estado_civil',
            'renda',
            'usuario_responsavel',
            'status',
        ]);
        if (array_filter($clienteData)) {
            $registro->cliente()->update($clienteData);
        }

        return response()->json($registro->load('cliente'));
    }

    public function destroy(int $id)
    {
        PessoaJuridica::query()->whereKey($id)->delete();
        return response()->noContent();
    }

    private function resolveCliente(Request $request, string $tipo): int
    {
        $clienteId = $request->input('cliente_id');
        if ($clienteId) {
            return (int) $clienteId;
        }

        $payload = $request->only([
            'nome_razao',
            'cpf_cnpj',
            'data_nascimento',
            'estado_civil',
            'renda',
            'usuario_responsavel',
            'status',
        ]);

        if (empty($payload['nome_razao']) && $request->filled('razao_social')) {
            $payload['nome_razao'] = $request->input('razao_social');
        }
        if (empty($payload['cpf_cnpj']) && $request->filled('cnpj')) {
            $payload['cpf_cnpj'] = $request->input('cnpj');
        }

        $payload['tipo_cliente'] = $tipo;

        $cliente = Cliente::query()->create($payload);

        return (int) $cliente->id;
    }

    private function normalizeEstadoCivil(?string $value): ?string
    {
        if (!$value) {
            return null;
        }
        $normalized = strtoupper(trim($value));
        $normalized = str_replace(' ', '_', $normalized);
        $allowed = ['CASADO', 'SOLTEIRO', 'DIVORCIADO', 'UNIAO_ESTAVEL', 'VIUVO'];
        return in_array($normalized, $allowed, true) ? $normalized : null;
    }

    private function storeEndereco(int $clienteId, Request $request): void
    {
        $payload = [
            'cliente_id' => $clienteId,
            'tipo_endereco' => 'EMPRESA',
            'cep' => $request->input('cep'),
            'logradouro' => $request->input('endereco'),
            'numero' => $request->input('numero'),
            'bairro' => $request->input('bairro'),
            'cidade' => $request->input('cidade'),
            'estado' => $request->input('estado'),
            'complemento' => $request->input('complemento'),
        ];

        $hasData = false;
        foreach ($payload as $key => $value) {
            if ($key === 'cliente_id' || $key === 'tipo_endereco') {
                continue;
            }
            if ($value !== null && $value !== '') {
                $hasData = true;
                break;
            }
        }
        if (!$hasData) {
            return;
        }

        DB::table('enderecos')->insert($payload);
    }

    private function ensureConjuge(?string $estadoCivilSocio, Request $request): void
    {
        if (!$estadoCivilSocio || !in_array($estadoCivilSocio, self::ESTADO_CIVIL_CASADO, true)) {
            return;
        }

        $nome = trim((string) $request->input('conjuge_nome'));
        $cpf = preg_replace('/\D+/', '', (string) $request->input('conjuge_cpf'));
        $dataNascimento = $request->input('conjuge_data_nascimento');

        if ($nome === '' || $cpf === '') {
            abort(422, 'Informe nome e CPF do conjuge.');
        }

        $cliente = Cliente::query()->where('cpf_cnpj', $cpf)->first();
        if (!$cliente) {
            $cliente = Cliente::query()->create([
                'tipo_cliente' => 'PF',
                'nome_razao' => $nome,
                'cpf_cnpj' => $cpf,
                'data_nascimento' => $dataNascimento ?: null,
                'estado_civil' => 'CASADO',
                'status' => 'lead',
            ]);
        }

        if (!PessoaFisica::query()->where('cliente_id', $cliente->id)->exists()) {
            PessoaFisica::query()->create([
                'cliente_id' => $cliente->id,
            ]);
        }
    }
}
