<?php

namespace App\Services;

use App\Models\AgendaEvento;
use App\Models\AgendaEventoLembrete;
use App\Models\Usuario;
use Carbon\Carbon;

class AgendaService
{
    public function isAdmin(?Usuario $user): bool
    {
        return $user && $user->role === 'admin';
    }

    public function canView(Usuario $user, AgendaEvento $evento): bool
    {
        if ($this->isAdmin($user)) {
            return true;
        }

        if ($evento->visibilidade === 'global') {
            return true;
        }

        if ($evento->owner_id && (int) $evento->owner_id === (int) $user->id) {
            return true;
        }

        return $evento->participantes()
            ->where('usuario_id', $user->id)
            ->exists();
    }

    public function canEdit(Usuario $user, AgendaEvento $evento): bool
    {
        if ($this->isAdmin($user)) {
            return true;
        }

        return $evento->owner_id && (int) $evento->owner_id === (int) $user->id;
    }

    public function renderSmsTemplate(string $template, array $data): string
    {
        $replacements = [
            '{titulo}' => $data['titulo'] ?? '',
            '{inicio}' => $data['inicio'] ?? '',
            '{fim}' => $data['fim'] ?? '',
            '{local}' => $data['local'] ?? '',
            '{cliente}' => $data['cliente'] ?? '',
            '{link}' => $data['link'] ?? '',
            '{responsavel}' => $data['responsavel'] ?? '',
        ];

        $message = strtr($template, $replacements);
        $lines = preg_split('/\r?\n/', $message) ?: [];
        $filtered = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '') {
                continue;
            }
            if (str_ends_with($line, ':')) {
                continue;
            }
            $filtered[] = $line;
        }

        return implode("\n", $filtered);
    }

    public function syncReminders(AgendaEvento $evento): void
    {
        AgendaEventoLembrete::query()
            ->where('evento_id', $evento->id)
            ->whereNull('enviado_em')
            ->delete();

        $minutes = (int) $evento->lembrete_minutos;
        if ($minutes <= 0) {
            return;
        }

        $sendAt = $evento->inicio_em instanceof Carbon
            ? $evento->inicio_em->copy()->subMinutes($minutes)
            : Carbon::parse($evento->inicio_em)->subMinutes($minutes);

        if ($evento->lembrete_sms_usuario && $evento->usuario_telefone) {
            AgendaEventoLembrete::query()->create([
                'evento_id' => $evento->id,
                'destinatario_tipo' => 'usuario',
                'destinatario_id' => $evento->owner_id,
                'destinatario_telefone' => $evento->usuario_telefone,
                'enviar_em' => $sendAt,
                'status' => 'pendente',
            ]);
        }

        if ($evento->lembrete_sms_cliente && $evento->cliente_telefone) {
            AgendaEventoLembrete::query()->create([
                'evento_id' => $evento->id,
                'destinatario_tipo' => 'cliente',
                'destinatario_id' => $evento->cliente_id,
                'destinatario_telefone' => $evento->cliente_telefone,
                'enviar_em' => $sendAt,
                'status' => 'pendente',
            ]);
        }
    }
}
