import { reactive } from 'vue';

const API_BASE = import.meta.env.VITE_API_BASE_URL || '/adm_capital/api';

function getCookie(name) {
  const value = `; ${document.cookie}`;
  const parts = value.split(`; ${name}=`);
  if (parts.length !== 2) {
    return null;
  }
  return decodeURIComponent(parts.pop().split(';').shift());
}

async function ensureCsrf() {
  await fetch(`${API_BASE}/csrf`, {
    method: 'GET',
    credentials: 'include',
  });
}

function buildHeaders() {
  const headers = {
    'Content-Type': 'application/json',
    'X-Requested-With': 'XMLHttpRequest',
  };
  const token = getCookie('XSRF-TOKEN');
  if (token) {
    headers['X-XSRF-TOKEN'] = token;
  }
  return headers;
}

export const authStore = reactive({
  user: null,
  isAuthenticated: false,
  initialized: false,
  loading: false,
  error: null,
});

export async function initAuth() {
  if (authStore.initialized) {
    return;
  }

  authStore.loading = true;
  authStore.error = null;

  try {
    const response = await fetch(`${API_BASE}/me`, {
      credentials: 'include',
      headers: buildHeaders(),
    });

    if (response.ok) {
      const data = await response.json();
      authStore.user = data.user;
      authStore.isAuthenticated = true;
    } else {
      authStore.user = null;
      authStore.isAuthenticated = false;
    }
  } catch (error) {
    authStore.error = 'Falha ao validar sessao.';
  } finally {
    authStore.loading = false;
    authStore.initialized = true;
  }
}

export async function login(email, password) {
  authStore.loading = true;
  authStore.error = null;

  try {
    await ensureCsrf();
    const response = await fetch(`${API_BASE}/login`, {
      method: 'POST',
      credentials: 'include',
      headers: buildHeaders(),
      body: JSON.stringify({ email, password }),
    });

    if (!response.ok) {
      const data = await response.json().catch(() => ({}));
      authStore.error = data.message || 'Credenciais invalidas.';
      authStore.isAuthenticated = false;
      authStore.user = null;
      return false;
    }

    const data = await response.json();
    authStore.user = data.user;
    authStore.isAuthenticated = true;
    return true;
  } catch (error) {
    authStore.error = 'Falha ao conectar com o servidor.';
    authStore.isAuthenticated = false;
    authStore.user = null;
    return false;
  } finally {
    authStore.loading = false;
  }
}

export async function logout() {
  authStore.loading = true;
  authStore.error = null;

  try {
    await ensureCsrf();
    await fetch(`${API_BASE}/logout`, {
      method: 'POST',
      credentials: 'include',
      headers: buildHeaders(),
    });
  } finally {
    authStore.user = null;
    authStore.isAuthenticated = false;
    authStore.loading = false;
  }
}
