<?php

namespace App\Http\Controllers;

use App\Models\PropostaCredito;
use App\Models\PropostaDocumento;
use App\Models\PropostaDocumentoValidacao;
use App\Services\DoclingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class ComplianceController extends Controller
{
    public function __construct(private DoclingService $docling)
    {
    }

    public function resumo(PropostaCredito $proposta)
    {
        $rows = PropostaDocumentoValidacao::query()
            ->where('proposta_id', $proposta->id)
            ->orderByDesc('id')
            ->get();

        if ($rows->isEmpty()) {
            return response()->json([
                'status' => 'pendente',
                'total' => 0,
                'aprovados' => 0,
                'divergentes' => 0,
                'pendentes' => 0,
            ]);
        }

        $latestByDoc = $rows->groupBy('documento_id')->map(function ($list) {
            return $list->first();
        });

        $aprovados = $latestByDoc->where('status', 'aprovado')->count();
        $divergentes = $latestByDoc->where('status', 'divergente')->count();
        $pendentes = $latestByDoc->where('status', 'pendente')->count();

        $status = 'aprovado';
        if ($divergentes > 0) {
            $status = 'divergente';
        } elseif ($pendentes > 0) {
            $status = 'pendente';
        }

        return response()->json([
            'status' => $status,
            'total' => $latestByDoc->count(),
            'aprovados' => $aprovados,
            'divergentes' => $divergentes,
            'pendentes' => $pendentes,
        ]);
    }

    public function list(PropostaCredito $proposta)
    {
        $rows = PropostaDocumentoValidacao::query()
            ->where('proposta_id', $proposta->id)
            ->orderByDesc('id')
            ->get()
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'proposta_id' => $item->proposta_id,
                    'documento_id' => $item->documento_id,
                    'status' => $item->status,
                    'score' => $item->score,
                    'campos' => $this->decodeJson($item->campos_json),
                    'alertas' => $this->decodeJson($item->alertas_json),
                    'extraido' => $this->decodeJson($item->extraido_json),
                    'created_at' => $item->created_at,
                ];
            });

        return response()->json([
            'data' => $rows,
        ]);
    }

    public function validarProposta(PropostaCredito $proposta)
    {
        $documentos = PropostaDocumento::query()
            ->where('proposta_id', $proposta->id)
            ->get();

        if ($documentos->isEmpty()) {
            return response()->json(['message' => 'Nenhum documento encontrado.'], 404);
        }

        $resultados = [];
        foreach ($documentos as $documento) {
            $resultados[] = $this->validarDocumentoInterno($proposta, $documento);
        }

        return response()->json([
            'message' => 'Validacao concluida.',
            'resultados' => $resultados,
        ]);
    }

    public function validarDocumento(PropostaCredito $proposta, PropostaDocumento $documento)
    {
        if ((int) $documento->proposta_id !== (int) $proposta->id) {
            return response()->json(['message' => 'Documento nao pertence a proposta.'], 422);
        }

        $resultado = $this->validarDocumentoInterno($proposta, $documento);
        return response()->json($resultado);
    }

    private function validarDocumentoInterno(PropostaCredito $proposta, PropostaDocumento $documento): array
    {
        $filePath = $this->resolveDocumentoPath($documento);
        if (!$filePath) {
            return [
                'ok' => false,
                'message' => 'Arquivo nao encontrado.',
            ];
        }

        $extraction = $this->docling->extractFromPath($filePath);
        if (!($extraction['ok'] ?? false)) {
            return [
                'ok' => false,
                'message' => $extraction['error'] ?? 'Falha ao extrair documento.',
            ];
        }

        $text = (string) ($extraction['text'] ?? '');
        $extraido = $this->extractFieldsFromText($text);
        $referencia = $this->buildReferenceData($proposta);
        $comparacao = $this->compareFields($referencia, $extraido);

        $status = $comparacao['status'];
        $score = $comparacao['score'];

        $validacao = PropostaDocumentoValidacao::query()->create([
            'proposta_id' => $proposta->id,
            'documento_id' => $documento->id,
            'status' => $status,
            'score' => $score,
            'campos_json' => json_encode($comparacao['campos']),
            'alertas_json' => json_encode($comparacao['alertas']),
            'extraido_json' => json_encode($extraido),
            'texto_extraido' => $text,
        ]);

        $documento->status_validacao = $status === 'aprovado' ? 'validado' : 'pendente';
        $documento->observacao = $comparacao['alertas'] ? implode(' | ', $comparacao['alertas']) : null;
        $documento->save();

        return [
            'ok' => true,
            'validacao_id' => $validacao->id,
            'status' => $status,
            'score' => $score,
            'campos' => $comparacao['campos'],
            'alertas' => $comparacao['alertas'],
        ];
    }

    private function resolveDocumentoPath(PropostaDocumento $documento): ?string
    {
        $path = $documento->arquivo_path ?: $documento->arquivo;
        if (!$path) {
            return null;
        }
        $publicPath = base_path('../public/'.$path);
        return file_exists($publicPath) ? $publicPath : null;
    }

    private function buildReferenceData(PropostaCredito $proposta): array
    {
        $cliente = DB::table('clientes')->where('id', $proposta->cliente_id)->first();
        $nome = $cliente->nome_razao ?? $cliente->nome ?? '';
        $cpfCnpj = $cliente->cpf_cnpj ?? '';
        $renda = $cliente->renda ?? null;

        $rg = null;
        if (Schema::hasTable('pessoas_fisicas')) {
            $rgColumn = $this->firstExistingColumn('pessoas_fisicas', ['rg', 'documento_rg', 'rg_numero']);
            if ($rgColumn) {
                $rg = DB::table('pessoas_fisicas')->where('cliente_id', $cliente->id)->value($rgColumn);
            }
        }

        $endereco = null;
        if (Schema::hasTable('enderecos')) {
            $row = DB::table('enderecos')
                ->where('cliente_id', $cliente->id)
                ->orderByDesc('id')
                ->first();
            if ($row) {
                $endereco = trim(implode(' ', array_filter([
                    $row->logradouro ?? null,
                    $row->numero ?? null,
                    $row->bairro ?? null,
                    $row->cidade ?? null,
                    $row->uf ?? null,
                    $row->cep ?? null,
                ])));
            }
        }

        return [
            'nome' => $nome,
            'cpf' => $cpfCnpj,
            'rg' => $rg,
            'renda' => $renda,
            'endereco' => $endereco,
        ];
    }

    private function extractFieldsFromText(string $text): array
    {
        $lines = preg_split('/\r?\n/', $text) ?: [];

        $cpfList = [];
        preg_match_all('/\b\d{3}\.?\d{3}\.?\d{3}-?\d{2}\b/', $text, $cpfMatches);
        foreach ($cpfMatches[0] ?? [] as $cpf) {
            $cpfList[] = $this->normalizeDigits($cpf);
        }

        $rgList = [];
        preg_match_all('/\b\d{1,2}\.?\d{3}\.?\d{3}-?[0-9Xx]\b/', $text, $rgMatches);
        foreach ($rgMatches[0] ?? [] as $rg) {
            $rgList[] = $this->normalizeDigits($rg);
        }

        $nomeList = [];
        foreach ($lines as $line) {
            $normalized = $this->normalizeText($line);
            if (str_contains($normalized, 'nome')) {
                $parts = explode(':', $line, 2);
                if (count($parts) === 2) {
                    $value = trim($parts[1]);
                    if ($value !== '') {
                        $nomeList[] = $value;
                    }
                }
            }
        }

        $rendaList = [];
        preg_match_all('/\bR\\$\\s*([\\d\\.]+,\\d{2})\b/', $text, $rendaMatches);
        foreach ($rendaMatches[1] ?? [] as $value) {
            $rendaList[] = $this->parseMoney($value);
        }

        $enderecoList = [];
        foreach ($lines as $line) {
            $normalized = $this->normalizeText($line);
            if (str_contains($normalized, 'endereco') || str_contains($normalized, 'logradouro')) {
                $parts = explode(':', $line, 2);
                $value = count($parts) === 2 ? $parts[1] : $line;
                $value = trim($value);
                if ($value !== '') {
                    $enderecoList[] = $value;
                }
            }
        }

        return [
            'cpf' => array_values(array_unique(array_filter($cpfList))),
            'rg' => array_values(array_unique(array_filter($rgList))),
            'nome' => array_values(array_unique(array_filter($nomeList))),
            'renda' => array_values(array_filter($rendaList)),
            'endereco' => array_values(array_filter($enderecoList)),
        ];
    }

    private function compareFields(array $referencia, array $extraido): array
    {
        $campos = [];
        $alertas = [];

        $cpfRef = $this->normalizeDigits($referencia['cpf'] ?? '');
        $cpfStatus = $this->compareExact($cpfRef, $extraido['cpf'] ?? []);
        $campos['cpf'] = $cpfStatus;
        $this->pushAlert('CPF', $cpfStatus, $alertas);

        $rgRef = $this->normalizeDigits($referencia['rg'] ?? '');
        $rgStatus = $this->compareExact($rgRef, $extraido['rg'] ?? []);
        $campos['rg'] = $rgStatus;
        $this->pushAlert('RG', $rgStatus, $alertas);

        $nomeRef = trim((string) ($referencia['nome'] ?? ''));
        $nomeStatus = $this->compareSimilarity($nomeRef, $extraido['nome'] ?? []);
        $campos['nome'] = $nomeStatus;
        $this->pushAlert('Nome', $nomeStatus, $alertas);

        $rendaRef = $this->parseMoney($referencia['renda'] ?? null);
        $rendaStatus = $this->compareMoney($rendaRef, $extraido['renda'] ?? []);
        $campos['renda'] = $rendaStatus;
        $this->pushAlert('Renda', $rendaStatus, $alertas);

        $enderecoRef = trim((string) ($referencia['endereco'] ?? ''));
        $enderecoStatus = $this->compareAddress($enderecoRef, $extraido['endereco'] ?? []);
        $campos['endereco'] = $enderecoStatus;
        $this->pushAlert('Endereco', $enderecoStatus, $alertas);

        $status = 'aprovado';
        foreach ($campos as $item) {
            if ($item['status'] === 'divergente') {
                $status = 'divergente';
                break;
            }
            if ($item['status'] === 'pendente') {
                $status = 'pendente';
            }
        }

        $okCount = count(array_filter($campos, fn ($item) => $item['status'] === 'ok'));
        $score = count($campos) ? round(($okCount / count($campos)) * 100, 2) : null;

        return [
            'status' => $status,
            'score' => $score,
            'campos' => $campos,
            'alertas' => $alertas,
        ];
    }

    private function compareExact(string $reference, array $values): array
    {
        if ($reference === '') {
            return ['status' => 'pendente', 'valor' => null, 'motivo' => 'Referencia ausente'];
        }
        if (!$values) {
            return ['status' => 'pendente', 'valor' => $reference, 'motivo' => 'Nao encontrado no documento'];
        }
        foreach ($values as $value) {
            if ($reference === $this->normalizeDigits($value)) {
                return ['status' => 'ok', 'valor' => $reference, 'motivo' => null];
            }
        }
        return ['status' => 'divergente', 'valor' => $reference, 'motivo' => 'Valor divergente'];
    }

    private function compareSimilarity(string $reference, array $values): array
    {
        if ($reference === '') {
            return ['status' => 'pendente', 'valor' => null, 'motivo' => 'Referencia ausente'];
        }
        if (!$values) {
            return ['status' => 'pendente', 'valor' => $reference, 'motivo' => 'Nao encontrado no documento'];
        }
        $best = 0;
        foreach ($values as $value) {
            $score = $this->similarity($reference, $value);
            if ($score > $best) {
                $best = $score;
            }
        }
        if ($best >= 0.85) {
            return ['status' => 'ok', 'valor' => $reference, 'motivo' => null, 'score' => $best];
        }
        return ['status' => 'divergente', 'valor' => $reference, 'motivo' => 'Nome divergente', 'score' => $best];
    }

    private function compareMoney(?float $reference, array $values): array
    {
        if (!$reference) {
            return ['status' => 'pendente', 'valor' => null, 'motivo' => 'Referencia ausente'];
        }
        if (!$values) {
            return ['status' => 'pendente', 'valor' => $reference, 'motivo' => 'Nao encontrado no documento'];
        }
        foreach ($values as $value) {
            $value = (float) $value;
            $tolerance = $reference * 0.1;
            if (abs($reference - $value) <= $tolerance) {
                return ['status' => 'ok', 'valor' => $reference, 'motivo' => null];
            }
        }
        return ['status' => 'divergente', 'valor' => $reference, 'motivo' => 'Renda divergente'];
    }

    private function compareAddress(string $reference, array $values): array
    {
        if ($reference === '') {
            return ['status' => 'pendente', 'valor' => null, 'motivo' => 'Referencia ausente'];
        }
        if (!$values) {
            return ['status' => 'pendente', 'valor' => $reference, 'motivo' => 'Nao encontrado no documento'];
        }
        $refTokens = $this->tokenize($reference);
        if (!$refTokens) {
            return ['status' => 'pendente', 'valor' => $reference, 'motivo' => 'Endereco sem tokens'];
        }
        foreach ($values as $value) {
            $docTokens = $this->tokenize($value);
            if (!$docTokens) {
                continue;
            }
            $intersection = array_intersect($refTokens, $docTokens);
            $ratio = count($intersection) / max(1, count($refTokens));
            if ($ratio >= 0.6) {
                return ['status' => 'ok', 'valor' => $reference, 'motivo' => null, 'score' => $ratio];
            }
        }
        return ['status' => 'divergente', 'valor' => $reference, 'motivo' => 'Endereco divergente'];
    }

    private function pushAlert(string $label, array $status, array &$alertas): void
    {
        if ($status['status'] === 'ok') {
            return;
        }
        $alertas[] = $label.': '.($status['motivo'] ?? 'Nao validado');
    }

    private function normalizeDigits(?string $value): string
    {
        return preg_replace('/\D+/', '', (string) $value);
    }

    private function normalizeText(string $value): string
    {
        $text = mb_strtolower($value, 'UTF-8');
        $text = iconv('UTF-8', 'ASCII//TRANSLIT', $text);
        $text = preg_replace('/[^a-z0-9\\s:]+/i', ' ', $text);
        return trim(preg_replace('/\s+/', ' ', $text));
    }

    private function tokenize(string $value): array
    {
        $text = $this->normalizeText($value);
        $parts = preg_split('/\s+/', $text) ?: [];
        return array_values(array_filter($parts, fn ($part) => strlen($part) > 2));
    }

    private function similarity(string $a, string $b): float
    {
        $a = $this->normalizeText($a);
        $b = $this->normalizeText($b);
        if ($a === '' || $b === '') {
            return 0.0;
        }
        $distance = levenshtein($a, $b);
        $maxLen = max(strlen($a), strlen($b));
        if ($maxLen === 0) {
            return 0.0;
        }
        return 1 - ($distance / $maxLen);
    }

    private function parseMoney($value): ?float
    {
        if ($value === null || $value === '') {
            return null;
        }
        $text = preg_replace('/[^0-9,\\.]/', '', (string) $value);
        if ($text === '') {
            return null;
        }
        $text = str_replace('.', '', $text);
        $text = str_replace(',', '.', $text);
        if (!is_numeric($text)) {
            return null;
        }
        return (float) $text;
    }

    private function firstExistingColumn(string $table, array $candidates): ?string
    {
        if (!Schema::hasTable($table)) {
            return null;
        }
        $columns = Schema::getColumnListing($table);
        foreach ($candidates as $candidate) {
            if (in_array($candidate, $columns, true)) {
                return $candidate;
            }
        }
        return null;
    }

    private function decodeJson(?string $value): array
    {
        if (!$value) {
            return [];
        }
        $data = json_decode($value, true);
        return is_array($data) ? $data : [];
    }
}
