<?php

namespace App\Http\Controllers;

use App\Models\Lead;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;

class ExternalFormController extends Controller
{
    private const EXTERNAL_CONNECTION = 'external_mysql';
    private const DOCUMENTS_TABLE = 'simulacoes_documentos';
    private const UPLOAD_DIR = '../formulario/api/uploads';

    private const FORM_MAP = [
        'credito-imobiliario' => [
            'table' => 'simulacoes_imobiliarias',
            'source' => 'simulacoes_imobiliarias',
            'tipo_solicitacao' => 'HOME EQUITY',
        ],
        'consignado' => [
            'table' => 'simulacoes_consignado',
            'source' => 'simulacoes_consignado',
            'tipo_solicitacao' => 'CONSIGNADO',
        ],
        'auto-equity-pf' => [
            'table' => 'simulacoes_auto_equity_pf',
            'source' => 'simulacoes_auto_equity_pf',
            'tipo_solicitacao' => 'AUTO EQUITY PF',
        ],
        'auto-equity-pj' => [
            'table' => 'simulacoes_auto_equity_pj',
            'source' => 'simulacoes_auto_equity_pj',
            'tipo_solicitacao' => 'AUTO EQUITY PJ',
        ],
        'financiamento-veiculo' => [
            'table' => 'simulacoes_financiamento_veiculo',
            'source' => 'simulacoes_financiamento_veiculo',
            'tipo_solicitacao' => 'FINANCIAMENTO VEICULO',
        ],
        'consorcio-automoveis' => [
            'table' => 'simulacoes_consorcio',
            'source' => 'simulacoes_consorcio',
            'tipo_solicitacao' => 'CONSORCIO AUTOMOVEIS',
        ],
        'financiamento-imobiliario-pf' => [
            'table' => 'simulacoes_financiamento_imobiliario_pf',
            'source' => 'simulacoes_financiamento_imobiliario_pf',
            'tipo_solicitacao' => 'FINANCIAMENTO IMOBILIARIO',
        ],
        'credito-construcao' => [
            'table' => 'simulacoes_credito_construcao',
            'source' => 'simulacoes_credito_construcao',
            'tipo_solicitacao' => 'CREDITO CONSTRUCAO',
        ],
        'credito-condominio-pj' => [
            'table' => 'simulacoes_credito_condominio',
            'source' => 'simulacoes_credito_condominio',
            'tipo_solicitacao' => 'CREDITO CONDOMINIO',
        ],
        'credito-aval-digital-fgi' => [
            'table' => 'simulacoes_credito_aval_digital_fgi',
            'source' => 'simulacoes_credito_aval_digital_fgi',
            'tipo_solicitacao' => 'CREDITO AVAL DIGITAL FGI',
        ],
        'creditos-pj-diversos' => [
            'table' => 'simulacoes_creditos_pj_diversos',
            'source' => 'simulacoes_creditos_pj_diversos',
            'tipo_solicitacao' => 'CREDITOS PJ DIVERSOS',
        ],
    ];

    public function upload(Request $request)
    {
        if (!$request->hasFile('documentos')) {
            return response()->json(['message' => 'Nenhum arquivo enviado.'], 422);
        }

        $files = $request->file('documentos');
        $files = is_array($files) ? $files : [$files];

        if (count($files) > 5) {
            return response()->json(['message' => 'Quantidade maxima de arquivos excedida.'], 422);
        }

        $allowed = ['pdf', 'jpg', 'jpeg', 'png'];
        $stored = [];
        $uploadDir = base_path(self::UPLOAD_DIR);

        if (!is_dir($uploadDir)) {
            @mkdir($uploadDir, 0750, true);
        }

        foreach ($files as $file) {
            if (!$file->isValid()) {
                return response()->json(['message' => 'Erro ao carregar arquivo.'], 422);
            }
            if ($file->getSize() > 5 * 1024 * 1024) {
                return response()->json(['message' => 'Arquivo maior que o limite permitido.'], 422);
            }

            $ext = strtolower($file->getClientOriginalExtension());
            if (!in_array($ext, $allowed, true)) {
                return response()->json(['message' => 'Extensao de arquivo nao permitida.'], 422);
            }

            $safeName = preg_replace('/[^A-Za-z0-9._-]/', '_', $file->getClientOriginalName());
            $unique = date('YmdHis') . '_' . bin2hex(random_bytes(4)) . '_' . $safeName;
            $file->move($uploadDir, $unique);
            $stored[] = $unique;
        }

        return response()->json(['success' => true, 'files' => $stored], 201);
    }

    public function store(Request $request, string $form)
    {
        if (!isset(self::FORM_MAP[$form])) {
            return response()->json(['message' => 'Formulario nao encontrado.'], 404);
        }

        $config = self::FORM_MAP[$form];
        $table = $config['table'];
        $source = $config['source'];
        $tipoSolicitacao = $config['tipo_solicitacao'];

        $columns = Schema::connection(self::EXTERNAL_CONNECTION)->getColumnListing($table);
        $payload = $request->all();
        $insert = [];

        foreach ($columns as $column) {
            if (array_key_exists($column, $payload)) {
                $insert[$column] = $payload[$column];
            }
        }

        if (in_array('origem', $columns, true) && empty($insert['origem'])) {
            $insert['origem'] = 'adm_capital/'.$form;
        }
        if (in_array('ip', $columns, true) && empty($insert['ip'])) {
            $insert['ip'] = $request->ip();
        }
        if (in_array('user_agent', $columns, true) && empty($insert['user_agent'])) {
            $ua = (string) $request->userAgent();
            $insert['user_agent'] = Str::limit($ua, 255, '');
        }
        if (in_array('created_at', $columns, true) && empty($insert['created_at'])) {
            $insert['created_at'] = now();
        }

        if (in_array('valor_emprestimo', $columns, true) && empty($insert['valor_emprestimo'])) {
            if (!empty($insert['valor_veiculo'])) {
                $insert['valor_emprestimo'] = $insert['valor_veiculo'];
            } elseif (!empty($insert['valor_imovel'])) {
                $insert['valor_emprestimo'] = $insert['valor_imovel'];
            }
        }

        $externalId = DB::connection(self::EXTERNAL_CONNECTION)->table($table)->insertGetId($insert);

        $docs = $request->input('documentos', []);
        if (is_array($docs) && $docs) {
            foreach ($docs as $doc) {
                if (!$doc) {
                    continue;
                }
                DB::connection(self::EXTERNAL_CONNECTION)->table(self::DOCUMENTS_TABLE)->insert([
                    'simulacao_id' => $externalId,
                    'arquivo' => $doc,
                    'tipo' => null,
                    'created_at' => now(),
                ]);
            }
        }

        $leadExists = Lead::query()
            ->where('external_source', $source)
            ->where('external_id', $externalId)
            ->exists();

        if (!$leadExists) {
            $lead = Lead::query()->create([
                'nome' => $this->resolveLeadName($payload),
                'email' => $payload['email'] ?? null,
                'telefone' => $payload['telefone'] ?? null,
                'origem' => $insert['origem'] ?? 'adm_capital',
                'status' => 'novo',
                'responsavel_id' => auth()->id(),
                'external_source' => $source,
                'external_id' => $externalId,
                'validated_at' => now(),
                'tipo_solicitacao' => $tipoSolicitacao,
            ]);
        }

        return response()->json([
            'success' => true,
            'external_id' => $externalId,
        ], 201);
    }

    private function resolveLeadName(array $payload): string
    {
        $candidates = [
            'nome',
            'nome_contato',
            'razao_social',
            'empresa_nome',
            'condominio_nome',
            'contato',
            'sindico_nome',
        ];
        foreach ($candidates as $field) {
            if (!empty($payload[$field])) {
                return (string) $payload[$field];
            }
        }
        return 'Lead';
    }
}
