<?php

namespace App\Http\Controllers;

use App\Models\Cliente;
use App\Models\Configuracao;
use App\Models\Lead;
use App\Models\LeadAccessLog;
use App\Models\PessoaFisica;
use App\Models\PessoaJuridica;
use App\Models\Usuario;
use App\Jobs\SendLeadAssignedNotifications;
use App\Services\LeadService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Schema;

class LeadController extends Controller
{
    private const STATUS = ['novo', 'contato', 'qualificacao', 'proposta', 'convertido', 'perdido'];
    private const DEFAULT_SMS_TEXTO_LEAD = "Novo lead #{id}\nNome: {nome}\nTelefone: {telefone}\nEmail: {email}\nOrigem: {origem}\nTipo: {tipo_solicitacao}\nResponsavel: {responsavel}";
    private const EXTERNAL_SOURCES = [
        'simulacoes_imobiliarias' => [
            'table' => 'simulacoes_imobiliarias',
            'tipo_solicitacao' => 'HOME EQUITY',
        ],
        'simulacoes_consignado' => [
            'table' => 'simulacoes_consignado',
            'tipo_solicitacao' => 'CONSIGNADO',
        ],
        'simulacoes_auto_equity_pf' => [
            'table' => 'simulacoes_auto_equity_pf',
            'tipo_solicitacao' => 'AUTO EQUITY PF',
        ],
        'simulacoes_auto_equity_pj' => [
            'table' => 'simulacoes_auto_equity_pj',
            'tipo_solicitacao' => 'AUTO EQUITY PJ',
        ],
        'simulacoes_financiamento_veiculo' => [
            'table' => 'simulacoes_financiamento_veiculo',
            'tipo_solicitacao' => 'FINANCIAMENTO VEICULO',
        ],
        'simulacoes_consorcio' => [
            'table' => 'simulacoes_consorcio',
            'tipo_solicitacao' => 'CONSORCIO AUTOMOVEIS',
        ],
        'simulacoes_financiamento_imobiliario_pf' => [
            'table' => 'simulacoes_financiamento_imobiliario_pf',
            'tipo_solicitacao' => 'FINANCIAMENTO IMOBILIARIO',
        ],
        'simulacoes_credito_construcao' => [
            'table' => 'simulacoes_credito_construcao',
            'tipo_solicitacao' => 'CREDITO CONSTRUCAO',
        ],
        'simulacoes_credito_condominio' => [
            'table' => 'simulacoes_credito_condominio',
            'tipo_solicitacao' => 'CREDITO CONDOMINIO',
        ],
        'simulacoes_credito_aval_digital_fgi' => [
            'table' => 'simulacoes_credito_aval_digital_fgi',
            'tipo_solicitacao' => 'CREDITO AVAL DIGITAL FGI',
        ],
        'simulacoes_creditos_pj_diversos' => [
            'table' => 'simulacoes_creditos_pj_diversos',
            'tipo_solicitacao' => 'CREDITOS PJ DIVERSOS',
        ],
    ];
    private const EXTERNAL_CONNECTION = 'external_mysql';
    private const EXTERNAL_DOCUMENTS_TABLE = 'simulacoes_documentos';
    private const TIPO_SOLICITACAO = [
        'HOME EQUITY',
        'CONSIGNADO',
        'AUTO EQUITY PF',
        'AUTO EQUITY PJ',
        'FINANCIAMENTO VEICULO',
        'CONSORCIO AUTOMOVEIS',
        'FINANCIAMENTO IMOBILIARIO',
        'CREDITO CONSTRUCAO',
        'CREDITO CONDOMINIO',
        'CREDITO AVAL DIGITAL FGI',
        'CREDITOS PJ DIVERSOS',
    ];

    public function __construct(private LeadService $service)
    {
    }

    public function index(Request $request)
    {
        $user = $request->user();

        $query = Lead::query()
            ->with(['responsavel', 'pessoaFisica', 'pessoaJuridica']);

        if (!$this->canViewAllLeads($user)) {
            $query->where('responsavel_id', $user?->id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->string('status')->toString());
        }
        if ($request->filled('tipo_solicitacao')) {
            $query->where('tipo_solicitacao', $request->string('tipo_solicitacao')->toString());
        }
        if ($request->filled('q')) {
            $term = trim($request->string('q')->toString());
            if ($term !== '') {
                $query->where(function ($builder) use ($term) {
                    $builder->where('nome', 'like', '%'.$term.'%')
                        ->orWhere('email', 'like', '%'.$term.'%')
                        ->orWhere('telefone', 'like', '%'.$term.'%');
                });
            }
        }

        $sortBy = $this->resolveLeadSortBy($request->query('sort_by'));
        $sortDir = $this->resolveSortDir($request->query('sort_dir'));

        $query->orderBy($sortBy, $sortDir);

        $perPage = $this->resolvePerPage($request->query('per_page'));
        $leads = $query->paginate($perPage);

        return response()->json($leads);
    }

    public function create()
    {
        return response()->json(['message' => 'create']);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'nome' => ['required', 'string', 'max:255'],
            'email' => ['nullable', 'email', 'max:255'],
            'telefone' => ['nullable', 'string', 'max:30'],
            'origem' => ['nullable', 'string', 'max:120'],
            'tipo_solicitacao' => ['nullable', 'string', 'max:40'],
            'external_source' => ['nullable', 'string', 'max:80'],
            'external_id' => ['nullable', 'integer'],
            'validated_at' => ['nullable', 'date'],
            'status' => ['nullable', 'string', 'in:'.implode(',', self::STATUS)],
            'responsavel_id' => ['nullable', 'integer', 'exists:users,id'],
            'pessoa_fisica_id' => ['nullable', 'integer', 'exists:pessoas_fisicas,cliente_id'],
            'pessoa_juridica_id' => ['nullable', 'integer', 'exists:pessoas_juridicas,cliente_id'],
        ]);

        $user = $request->user();
        if (!$this->canViewAllLeads($user)) {
            $data['responsavel_id'] = $user?->id;
        }

        if (empty($data['status'])) {
            $data['status'] = 'novo';
        }

        $lead = Lead::query()->create($data);
        $this->dispatchLeadNotification($lead, $lead->responsavel_id);
        return response()->json($lead, 201);
    }

    public function show(Request $request, int $id)
    {
        $lead = Lead::query()
            ->with(['responsavel', 'pessoaFisica', 'pessoaJuridica'])
            ->findOrFail($id);

        $this->authorizeLeadAccess($lead, $request->user());
        $this->logLeadAccess($lead->id, $request);

        return response()->json($lead);
    }

    public function edit(int $id)
    {
        return response()->json(Lead::query()->findOrFail($id));
    }

    public function update(Request $request, int $id)
    {
        $lead = Lead::query()->findOrFail($id);
        $this->authorizeLeadAccess($lead, $request->user());
        $data = $request->validate([
            'nome' => ['sometimes', 'required', 'string', 'max:255'],
            'email' => ['nullable', 'email', 'max:255'],
            'telefone' => ['nullable', 'string', 'max:30'],
            'origem' => ['nullable', 'string', 'max:120'],
            'tipo_solicitacao' => ['nullable', 'string', 'max:40'],
            'external_source' => ['nullable', 'string', 'max:80'],
            'external_id' => ['nullable', 'integer'],
            'validated_at' => ['nullable', 'date'],
            'responsavel_id' => ['nullable', 'integer', 'exists:users,id'],
            'pessoa_fisica_id' => ['nullable', 'integer', 'exists:pessoas_fisicas,cliente_id'],
            'pessoa_juridica_id' => ['nullable', 'integer', 'exists:pessoas_juridicas,cliente_id'],
        ]);

        $lead->update($data);
        return response()->json($lead);
    }

    public function destroy(Request $request, int $id)
    {
        $lead = Lead::query()->findOrFail($id);
        $this->authorizeLeadAccess($lead, $request->user());
        $lead->delete();
        return response()->noContent();
    }

    public function updateStatus(Request $request, int $id)
    {
        $lead = Lead::query()->findOrFail($id);
        $this->authorizeLeadAccess($lead, $request->user());
        $data = $request->validate([
            'status' => ['required', 'string', 'in:'.implode(',', self::STATUS)],
        ]);

        try {
            $lead = $this->service->transition($lead, $data['status']);
        } catch (\InvalidArgumentException $exception) {
            return response()->json(['message' => $exception->getMessage()], 422);
        }

        return response()->json($lead);
    }

    public function assignResponsavel(Request $request, int $id)
    {
        $lead = Lead::query()->findOrFail($id);
        $data = $request->validate([
            'usuario_id' => ['required', 'integer', 'exists:users,id'],
        ]);
        $user = $request->user();
        $this->authorizeLeadAssignment($lead, $user, (int) $data['usuario_id']);

        $previous = $lead->responsavel_id;
        $lead = $this->service->assignResponsavel($lead, (int) $data['usuario_id']);

        if ($lead->responsavel_id && (int) $lead->responsavel_id !== (int) $previous) {
            $this->dispatchLeadNotification($lead, $lead->responsavel_id);
        }
        return response()->json($lead);
    }

    public function convertToProposta(Request $request, int $id)
    {
        $lead = Lead::query()->findOrFail($id);
        $this->authorizeLeadAccess($lead, $request->user());
        $lead = $this->service->convertToProposta($lead);
        return response()->json($lead);
    }

    public function externalIndex(Request $request)
    {
        $data = collect();
        $user = $request->user();
        $canViewAll = $this->canViewAllLeads($user);
        $userId = $user?->id;

        if (isset(self::EXTERNAL_SOURCES['simulacoes_imobiliarias'])) {
            $source = 'simulacoes_imobiliarias';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $this->ensureExternalDocumentsTable();
            } catch (\Throwable $exception) {
                logger()->warning('Falha ao garantir tabela de documentos externos.', [
                    'error' => $exception->getMessage(),
                ]);
            }

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda',
                        's.cep',
                        's.endereco',
                        's.numero',
                        's.bairro',
                        's.complemento',
                        's.cidade',
                        's.estado',
                        's.valor_imovel',
                        's.tipo_imovel',
                        's.propriedade',
                        's.metragem',
                        's.financiado',
                        's.saldo_financiado',
                        's.condominio',
                        's.valor_emprestimo',
                        's.prazo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $documentsById = $this->loadExternalDocuments($externalIds);
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));

            $data = $data->concat($items->map(function ($item) use ($documentsById, $source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $leadInfo = $leadByExternal[$externalId] ?? [];
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda,
                    'cep' => $item->cep,
                    'endereco' => $item->endereco,
                    'numero' => $item->numero,
                    'bairro' => $item->bairro,
                    'complemento' => $item->complemento,
                    'cidade' => $item->cidade,
                    'estado' => $item->estado,
                    'valor_imovel' => $item->valor_imovel,
                    'tipo_imovel' => $item->tipo_imovel,
                    'propriedade' => $item->propriedade,
                    'metragem' => $item->metragem,
                    'financiado' => $item->financiado,
                    'saldo_financiado' => $item->saldo_financiado,
                    'condominio' => $item->condominio,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => $item->prazo,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'documents' => $documentsById[$externalId] ?? [],
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_consignado'])) {
            $source = 'simulacoes_consignado';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda_mensal',
                        's.cep',
                        's.endereco',
                        's.numero',
                        's.bairro',
                        's.complemento',
                        's.cidade',
                        's.estado',
                        's.valor_emprestimo',
                        's.data_nascimento',
                        's.situacao_profissional',
                        's.motivo_emprestimo',
                        's.veiculo_nome_cliente',
                        's.placa',
                        's.renavam',
                        's.ano_fabricacao',
                        's.ano_modelo',
                        's.marca',
                        's.modelo',
                        's.pacote_acessorios',
                        's.versao',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));

            $data = $data->concat($items->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $leadInfo = $leadByExternal[$externalId] ?? [];
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda_mensal,
                    'renda_mensal' => $item->renda_mensal,
                    'cep' => $item->cep,
                    'endereco' => $item->endereco,
                    'numero' => $item->numero,
                    'bairro' => $item->bairro,
                    'complemento' => $item->complemento,
                    'cidade' => $item->cidade,
                    'estado' => $item->estado,
                    'valor_imovel' => null,
                    'tipo_imovel' => 'VEICULO',
                    'propriedade' => null,
                    'metragem' => null,
                    'financiado' => null,
                    'saldo_financiado' => null,
                    'condominio' => null,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'data_nascimento' => $item->data_nascimento,
                    'situacao_profissional' => $item->situacao_profissional,
                    'motivo_emprestimo' => $item->motivo_emprestimo,
                    'veiculo_nome_cliente' => $item->veiculo_nome_cliente,
                    'placa' => $item->placa,
                    'renavam' => $item->renavam,
                    'ano_fabricacao' => $item->ano_fabricacao,
                    'ano_modelo' => $item->ano_modelo,
                    'marca' => $item->marca,
                    'modelo' => $item->modelo,
                    'pacote_acessorios' => $item->pacote_acessorios,
                    'versao' => $item->versao,
                    'documents' => [],
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_auto_equity_pf'])) {
            $source = 'simulacoes_auto_equity_pf';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda',
                        's.valor_emprestimo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));
            $clienteIds = collect($clienteByDocument)->values()->unique();
            $propostaClienteIds = $this->loadPropostaClienteIds($clienteIds);

            $data = $data->concat($items->filter(function ($item) use ($clienteByDocument, $propostaClienteIds) {
                $document = $this->normalizeDocument($item->cpf ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                if (!$clienteId) {
                    return false;
                }
                return !in_array((int) $clienteId, $propostaClienteIds, true);
            })->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                $leadInfo = $leadByExternal[$externalId] ?? [];
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_auto_equity_pj'])) {
            $source = 'simulacoes_auto_equity_pj';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.empresa_nome',
                        's.empresa_cnpj',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.faturamento_12m',
                        's.valor_emprestimo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('empresa_cnpj'));
            $clienteIds = collect($clienteByDocument)->values()->unique();
            $propostaClienteIds = $this->loadPropostaClienteIds($clienteIds);

            $data = $data->concat($items->filter(function ($item) use ($clienteByDocument, $propostaClienteIds) {
                $document = $this->normalizeDocument($item->empresa_cnpj ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                if (!$clienteId) {
                    return false;
                }
                return !in_array((int) $clienteId, $propostaClienteIds, true);
            })->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->empresa_cnpj ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                $leadInfo = $leadByExternal[$externalId] ?? [];
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'nome' => $item->empresa_nome ?: $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->empresa_cnpj ?: $item->cpf,
                    'renda' => $item->faturamento_12m,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_financiamento_veiculo'])) {
            $source = 'simulacoes_financiamento_veiculo';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.cnpj',
                        's.tipo_pessoa',
                        's.renda',
                        's.faturamento_anual',
                        's.valor_emprestimo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $documents = $items->map(fn ($item) => $item->tipo_pessoa === 'PJ' ? $item->cnpj : $item->cpf);
            $clienteByDocument = $this->loadClientesByDocument($documents);
            $clienteIds = collect($clienteByDocument)->values()->unique();
            $propostaClienteIds = $this->loadPropostaClienteIds($clienteIds);

            $data = $data->concat($items->filter(function ($item) use ($clienteByDocument, $propostaClienteIds) {
                $documentValue = $item->tipo_pessoa === 'PJ' ? $item->cnpj : $item->cpf;
                $document = $this->normalizeDocument($documentValue ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                if (!$clienteId) {
                    return false;
                }
                return !in_array((int) $clienteId, $propostaClienteIds, true);
            })->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $documentValue = $item->tipo_pessoa === 'PJ' ? $item->cnpj : $item->cpf;
                $document = $this->normalizeDocument($documentValue ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                $leadInfo = $leadByExternal[$externalId] ?? [];
                $renda = $item->renda;
                if (!$renda && $item->tipo_pessoa === 'PJ') {
                    $renda = $item->faturamento_anual;
                }
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $documentValue,
                    'renda' => $renda,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_consorcio'])) {
            $source = 'simulacoes_consorcio';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda',
                        's.valor_emprestimo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));
            $clienteIds = collect($clienteByDocument)->values()->unique();
            $propostaClienteIds = $this->loadPropostaClienteIds($clienteIds);

            $data = $data->concat($items->filter(function ($item) use ($clienteByDocument, $propostaClienteIds) {
                $document = $this->normalizeDocument($item->cpf ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                if (!$clienteId) {
                    return false;
                }
                return !in_array((int) $clienteId, $propostaClienteIds, true);
            })->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                $leadInfo = $leadByExternal[$externalId] ?? [];
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_financiamento_imobiliario_pf'])) {
            $source = 'simulacoes_financiamento_imobiliario_pf';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda',
                        's.valor_emprestimo',
                        's.prazo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));
            $clienteIds = collect($clienteByDocument)->values()->unique();
            $propostaClienteIds = $this->loadPropostaClienteIds($clienteIds);

            $data = $data->concat($items->filter(function ($item) use ($clienteByDocument, $propostaClienteIds) {
                $document = $this->normalizeDocument($item->cpf ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                if (!$clienteId) {
                    return false;
                }
                return !in_array((int) $clienteId, $propostaClienteIds, true);
            })->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                $leadInfo = $leadByExternal[$externalId] ?? [];
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => $item->prazo,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_credito_construcao'])) {
            $source = 'simulacoes_credito_construcao';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.valor_emprestimo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));
            $clienteIds = collect($clienteByDocument)->values()->unique();
            $propostaClienteIds = $this->loadPropostaClienteIds($clienteIds);

            $data = $data->concat($items->filter(function ($item) use ($clienteByDocument, $propostaClienteIds) {
                $document = $this->normalizeDocument($item->cpf ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                if (!$clienteId) {
                    return false;
                }
                return !in_array((int) $clienteId, $propostaClienteIds, true);
            })->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                $leadInfo = $leadByExternal[$externalId] ?? [];
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => null,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_credito_condominio'])) {
            $source = 'simulacoes_credito_condominio';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.condominio_nome',
                        's.condominio_cnpj',
                        's.email',
                        's.telefone',
                        's.arrecadacao_mensal',
                        's.valor_emprestimo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('condominio_cnpj'));
            $clienteIds = collect($clienteByDocument)->values()->unique();
            $propostaClienteIds = $this->loadPropostaClienteIds($clienteIds);

            $data = $data->concat($items->filter(function ($item) use ($clienteByDocument, $propostaClienteIds) {
                $document = $this->normalizeDocument($item->condominio_cnpj ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                if (!$clienteId) {
                    return false;
                }
                return !in_array((int) $clienteId, $propostaClienteIds, true);
            })->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->condominio_cnpj ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                $leadInfo = $leadByExternal[$externalId] ?? [];
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'nome' => $item->condominio_nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->condominio_cnpj,
                    'renda' => $item->arrecadacao_mensal,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_credito_aval_digital_fgi'])) {
            $source = 'simulacoes_credito_aval_digital_fgi';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.razao_social',
                        's.cnpj',
                        's.email',
                        's.telefone',
                        's.faturamento_anual',
                        's.valor_solicitado',
                        's.valor_emprestimo',
                        's.prazo_meses',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cnpj'));
            $clienteIds = collect($clienteByDocument)->values()->unique();
            $propostaClienteIds = $this->loadPropostaClienteIds($clienteIds);

            $data = $data->concat($items->filter(function ($item) use ($clienteByDocument, $propostaClienteIds) {
                $document = $this->normalizeDocument($item->cnpj ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                if (!$clienteId) {
                    return false;
                }
                return !in_array((int) $clienteId, $propostaClienteIds, true);
            })->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cnpj ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                $leadInfo = $leadByExternal[$externalId] ?? [];
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'nome' => $item->razao_social,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cnpj,
                    'renda' => $item->faturamento_anual,
                    'valor_emprestimo' => $item->valor_emprestimo ?: $item->valor_solicitado,
                    'prazo' => $item->prazo_meses,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_creditos_pj_diversos'])) {
            $source = 'simulacoes_creditos_pj_diversos';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.razao_social',
                        's.cnpj',
                        's.email',
                        's.telefone',
                        's.faturamento_anual',
                        's.valor_proposta',
                        's.valor_emprestimo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cnpj'));
            $clienteIds = collect($clienteByDocument)->values()->unique();
            $propostaClienteIds = $this->loadPropostaClienteIds($clienteIds);

            $data = $data->concat($items->filter(function ($item) use ($clienteByDocument, $propostaClienteIds) {
                $document = $this->normalizeDocument($item->cnpj ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                if (!$clienteId) {
                    return false;
                }
                return !in_array((int) $clienteId, $propostaClienteIds, true);
            })->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cnpj ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                $leadInfo = $leadByExternal[$externalId] ?? [];
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'nome' => $item->razao_social,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cnpj,
                    'renda' => $item->faturamento_anual,
                    'valor_emprestimo' => $item->valor_emprestimo ?: $item->valor_proposta,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_credito_aval_digital_fgi'])) {
            $source = 'simulacoes_credito_aval_digital_fgi';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.razao_social',
                        's.cnpj',
                        's.email',
                        's.telefone',
                        's.faturamento_anual',
                        's.valor_solicitado',
                        's.valor_emprestimo',
                        's.prazo_meses',
                        's.segmento',
                        's.funcionarios',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cnpj'));

            $data = $data->concat($items->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cnpj ?? '');
                $leadInfo = $leadByExternal[$externalId] ?? [];
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'nome' => $item->razao_social ?: $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cnpj,
                    'renda' => $item->faturamento_anual,
                    'cep' => null,
                    'endereco' => null,
                    'numero' => null,
                    'bairro' => null,
                    'complemento' => null,
                    'cidade' => null,
                    'estado' => null,
                    'valor_imovel' => $item->valor_solicitado,
                    'tipo_imovel' => null,
                    'propriedade' => null,
                    'metragem' => null,
                    'financiado' => null,
                    'saldo_financiado' => null,
                    'condominio' => null,
                    'valor_emprestimo' => $item->valor_emprestimo ?: $item->valor_solicitado,
                    'prazo' => $item->prazo_meses,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'documents' => [],
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_creditos_pj_diversos'])) {
            $source = 'simulacoes_creditos_pj_diversos';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.razao_social',
                        's.cnpj',
                        's.email',
                        's.telefone',
                        's.tipo_operacao',
                        's.faturamento_anual',
                        's.valor_proposta',
                        's.valor_emprestimo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cnpj'));

            $data = $data->concat($items->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cnpj ?? '');
                $leadInfo = $leadByExternal[$externalId] ?? [];
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'nome' => $item->razao_social ?: $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cnpj,
                    'renda' => $item->faturamento_anual,
                    'cep' => null,
                    'endereco' => null,
                    'numero' => null,
                    'bairro' => null,
                    'complemento' => null,
                    'cidade' => null,
                    'estado' => null,
                    'valor_imovel' => $item->valor_proposta,
                    'tipo_imovel' => $item->tipo_operacao,
                    'tipo_operacao' => $item->tipo_operacao,
                    'propriedade' => null,
                    'metragem' => null,
                    'financiado' => null,
                    'saldo_financiado' => null,
                    'condominio' => null,
                    'valor_emprestimo' => $item->valor_emprestimo ?: $item->valor_proposta,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'documents' => [],
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_auto_equity_pf'])) {
            $source = 'simulacoes_auto_equity_pf';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda',
                        's.data_nascimento',
                        's.cep',
                        's.endereco',
                        's.numero',
                        's.bairro',
                        's.complemento',
                        's.cidade',
                        's.estado',
                        's.valor_emprestimo',
                        's.veiculo_tipo',
                        's.veiculo_marca',
                        's.veiculo_modelo',
                        's.veiculo_versao',
                        's.ano_fabricacao',
                        's.ano_modelo',
                        's.placa',
                        's.veiculo_nome',
                        's.alienacao',
                        's.saldo_devedor',
                        's.instituicao_saldo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));

            $data = $data->concat($items->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $leadInfo = $leadByExternal[$externalId] ?? [];
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda,
                    'cep' => $item->cep,
                    'endereco' => $item->endereco,
                    'numero' => $item->numero,
                    'bairro' => $item->bairro,
                    'complemento' => $item->complemento,
                    'cidade' => $item->cidade,
                    'estado' => $item->estado,
                    'valor_imovel' => $item->valor_emprestimo,
                    'tipo_imovel' => $item->veiculo_tipo ?: 'VEICULO',
                    'propriedade' => null,
                    'metragem' => null,
                    'financiado' => $item->alienacao,
                    'saldo_financiado' => $item->saldo_devedor,
                    'condominio' => null,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'data_nascimento' => $item->data_nascimento,
                    'placa' => $item->placa,
                    'ano_fabricacao' => $item->ano_fabricacao,
                    'ano_modelo' => $item->ano_modelo,
                    'marca' => $item->veiculo_marca,
                    'modelo' => $item->veiculo_modelo,
                    'versao' => $item->veiculo_versao,
                    'veiculo_nome_cliente' => $item->veiculo_nome,
                    'documents' => [],
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_auto_equity_pj'])) {
            $source = 'simulacoes_auto_equity_pj';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.empresa_nome',
                        's.empresa_cnpj',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.faturamento_12m',
                        's.cep',
                        's.endereco',
                        's.numero',
                        's.bairro',
                        's.complemento',
                        's.cidade',
                        's.estado',
                        's.valor_emprestimo',
                        's.veiculo_marca',
                        's.veiculo_modelo',
                        's.veiculo_versao',
                        's.ano_fabricacao',
                        's.ano_modelo',
                        's.placa',
                        's.veiculo_nome_empresa',
                        's.socio_nome',
                        's.socio_cpf',
                        's.socio_data_nascimento',
                        's.socio_renda',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));

            $data = $data->concat($items->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $leadInfo = $leadByExternal[$externalId] ?? [];
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'nome' => $item->empresa_nome ?: $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->empresa_cnpj ?: $item->cpf,
                    'renda' => $item->faturamento_12m,
                    'cep' => $item->cep,
                    'endereco' => $item->endereco,
                    'numero' => $item->numero,
                    'bairro' => $item->bairro,
                    'complemento' => $item->complemento,
                    'cidade' => $item->cidade,
                    'estado' => $item->estado,
                    'valor_imovel' => $item->valor_emprestimo,
                    'tipo_imovel' => 'VEICULO',
                    'propriedade' => null,
                    'metragem' => null,
                    'financiado' => null,
                    'saldo_financiado' => null,
                    'condominio' => null,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'data_nascimento' => $item->socio_data_nascimento,
                    'placa' => $item->placa,
                    'ano_fabricacao' => $item->ano_fabricacao,
                    'ano_modelo' => $item->ano_modelo,
                    'marca' => $item->veiculo_marca,
                    'modelo' => $item->veiculo_modelo,
                    'versao' => $item->veiculo_versao,
                    'veiculo_nome_cliente' => $item->veiculo_nome_empresa,
                    'documents' => [],
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_financiamento_veiculo'])) {
            $source = 'simulacoes_financiamento_veiculo';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.cnpj',
                        's.tipo_pessoa',
                        's.renda',
                        's.faturamento_anual',
                        's.condicao_veiculo',
                        's.tipo_veiculo',
                        's.marca',
                        's.modelo',
                        's.ano_fabricacao',
                        's.ano_modelo',
                        's.combustivel',
                        's.uf_licenciamento',
                        's.valor_veiculo',
                        's.valor_entrada',
                        's.valor_emprestimo',
                        's.data_nascimento',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $documents = $items->map(fn ($item) => $item->tipo_pessoa === 'PJ' ? $item->cnpj : $item->cpf);
            $clienteByDocument = $this->loadClientesByDocument($documents);

            $data = $data->concat($items->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $documentValue = $item->tipo_pessoa === 'PJ' ? $item->cnpj : $item->cpf;
                $document = $this->normalizeDocument($documentValue ?? '');
                $externalId = (int) $item->id;
                $leadInfo = $leadByExternal[$externalId] ?? [];
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                $renda = $item->renda;
                if (!$renda && $item->tipo_pessoa === 'PJ') {
                    $renda = $item->faturamento_anual;
                }
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $documentValue,
                    'renda' => $renda,
                    'cep' => null,
                    'endereco' => null,
                    'numero' => null,
                    'bairro' => null,
                    'complemento' => null,
                    'cidade' => null,
                    'estado' => $item->uf_licenciamento,
                    'valor_imovel' => $item->valor_veiculo,
                    'tipo_imovel' => $item->tipo_veiculo ?: 'VEICULO',
                    'propriedade' => null,
                    'metragem' => null,
                    'financiado' => $item->condicao_veiculo,
                    'saldo_financiado' => null,
                    'condominio' => null,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'data_nascimento' => $item->data_nascimento,
                    'ano_fabricacao' => $item->ano_fabricacao,
                    'ano_modelo' => $item->ano_modelo,
                    'marca' => $item->marca,
                    'modelo' => $item->modelo,
                    'documents' => [],
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_consorcio'])) {
            $source = 'simulacoes_consorcio';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda',
                        's.tipo_bem',
                        's.valor_bem',
                        's.valor_emprestimo',
                        's.lance_embutido',
                        's.lance_proprio',
                        's.valor_parcela_max',
                        's.prazo_desejado',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));

            $data = $data->concat($items->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $leadInfo = $leadByExternal[$externalId] ?? [];
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda,
                    'cep' => null,
                    'endereco' => null,
                    'numero' => null,
                    'bairro' => null,
                    'complemento' => null,
                    'cidade' => null,
                    'estado' => null,
                    'valor_imovel' => $item->valor_bem,
                    'tipo_imovel' => $item->tipo_bem,
                    'propriedade' => null,
                    'metragem' => null,
                    'financiado' => null,
                    'saldo_financiado' => null,
                    'condominio' => null,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'documents' => [],
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_financiamento_imobiliario_pf'])) {
            $source = 'simulacoes_financiamento_imobiliario_pf';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda',
                        's.data_nascimento',
                        's.tipo_imovel',
                        's.uf_imovel',
                        's.valor_imovel',
                        's.valor_financiamento',
                        's.valor_emprestimo',
                        's.prazo',
                        's.imovel_definido',
                        's.usar_fgts',
                        's.incluir_itbi',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));

            $data = $data->concat($items->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $leadInfo = $leadByExternal[$externalId] ?? [];
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda,
                    'cep' => null,
                    'endereco' => null,
                    'numero' => null,
                    'bairro' => null,
                    'complemento' => null,
                    'cidade' => null,
                    'estado' => $item->uf_imovel,
                    'valor_imovel' => $item->valor_imovel,
                    'tipo_imovel' => $item->tipo_imovel,
                    'propriedade' => null,
                    'metragem' => null,
                    'financiado' => $item->imovel_definido,
                    'saldo_financiado' => $item->valor_financiamento,
                    'condominio' => null,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => $item->prazo,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'data_nascimento' => $item->data_nascimento,
                    'documents' => [],
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_credito_construcao'])) {
            $source = 'simulacoes_credito_construcao';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.tipo_pessoa',
                        's.valor_proposta',
                        's.valor_emprestimo',
                        's.valor_imovel_acabado',
                        's.tipo_construcao',
                        's.percentual_construcao',
                        's.alvara',
                        's.art_rrt',
                        's.projeto_arquitetonico',
                        's.cep',
                        's.endereco',
                        's.numero',
                        's.bairro',
                        's.complemento',
                        's.cidade',
                        's.estado',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));

            $data = $data->concat($items->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $leadInfo = $leadByExternal[$externalId] ?? [];
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => null,
                    'cep' => $item->cep,
                    'endereco' => $item->endereco,
                    'numero' => $item->numero,
                    'bairro' => $item->bairro,
                    'complemento' => $item->complemento,
                    'cidade' => $item->cidade,
                    'estado' => $item->estado,
                    'valor_imovel' => $item->valor_imovel_acabado,
                    'tipo_imovel' => $item->tipo_construcao,
                    'propriedade' => null,
                    'metragem' => null,
                    'financiado' => null,
                    'saldo_financiado' => null,
                    'condominio' => null,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'documents' => [],
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_credito_condominio'])) {
            $source = 'simulacoes_credito_condominio';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.condominio_nome',
                        's.condominio_cnpj',
                        's.email',
                        's.telefone',
                        's.cep',
                        's.endereco',
                        's.cidade',
                        's.estado',
                        's.valor_imovel',
                        's.arrecadacao_mensal',
                        's.valor_credito',
                        's.valor_emprestimo',
                        's.finalidade',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('condominio_cnpj'));

            $data = $data->concat($items->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->condominio_cnpj ?? '');
                $leadInfo = $leadByExternal[$externalId] ?? [];
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'nome' => $item->condominio_nome ?: $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->condominio_cnpj,
                    'renda' => $item->arrecadacao_mensal,
                    'cep' => $item->cep,
                    'endereco' => $item->endereco,
                    'numero' => null,
                    'bairro' => null,
                    'complemento' => null,
                    'cidade' => $item->cidade,
                    'estado' => $item->estado,
                    'valor_imovel' => $item->valor_imovel,
                    'tipo_imovel' => 'CONDOMINIO',
                    'propriedade' => null,
                    'metragem' => null,
                    'financiado' => null,
                    'saldo_financiado' => null,
                    'condominio' => null,
                    'valor_emprestimo' => $item->valor_emprestimo ?: $item->valor_credito,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'documents' => [],
                ];
            }));
        }

        if (!$canViewAll && $userId) {
            $data = $data->filter(function ($item) use ($userId) {
                return (int) ($item['responsavel_id'] ?? 0) === (int) $userId;
            })->values();
        }

        $term = trim($request->string('q')->toString());
        if ($term !== '') {
            $term = mb_strtolower($term);
            $data = $data->filter(function ($item) use ($term) {
                $values = [
                    $item['nome'] ?? '',
                    $item['email'] ?? '',
                    $item['cpf'] ?? '',
                    $item['telefone'] ?? '',
                ];
                foreach ($values as $value) {
                    if ($value !== '' && str_contains(mb_strtolower((string) $value), $term)) {
                        return true;
                    }
                }
                return false;
            })->values();
        }

        $showImported = $request->boolean('show_imported', false);
        if (!$showImported) {
            $data = $data->filter(function ($item) {
                return empty($item['lead_id']);
            })->values();
        }

        $sortBy = $this->resolveExternalSortBy($request->query('sort_by'));
        $sortDir = $this->resolveSortDir($request->query('sort_dir'));
        if ($sortBy) {
            $data = $sortDir === 'asc'
                ? $data->sortBy(fn ($item) => $item[$sortBy] ?? null)
                : $data->sortByDesc(fn ($item) => $item[$sortBy] ?? null);
        } else {
            $data = $data->sortByDesc(function ($item) {
                return $item['created_at'] ?? '';
            });
        }
        $data = $data->values();

        $data = $data->map(function ($item) {
            $item['telefone'] = $this->normalizeTelefone($item['telefone'] ?? '');
            return $item;
        });

        if ($request->filled('per_page') || $request->filled('page')) {
            $perPage = $this->resolvePerPage($request->query('per_page'));
            $page = max(1, (int) $request->query('page', 1));
            $total = $data->count();
            $items = $data->slice(($page - 1) * $perPage, $perPage)->values();

            return response()->json([
                'data' => $items,
                'meta' => [
                    'current_page' => $page,
                    'per_page' => $perPage,
                    'total' => $total,
                    'last_page' => $perPage > 0 ? (int) ceil($total / $perPage) : 1,
                ],
            ]);
        }

        return response()->json(['data' => $data]);
    }

    public function availableForPropostas(Request $request)
    {
        $data = collect();
        $user = $request->user();
        $canViewAll = $this->canViewAllLeads($user);
        $userId = $user?->id;

        if (isset(self::EXTERNAL_SOURCES['simulacoes_imobiliarias'])) {
            $source = 'simulacoes_imobiliarias';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda',
                        's.valor_emprestimo',
                        's.prazo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));
            $clienteIds = collect($clienteByDocument)->values()->unique();
            $propostaClienteIds = $this->loadPropostaClienteIds($clienteIds);

            $data = $data->concat($items->filter(function ($item) use ($clienteByDocument, $propostaClienteIds) {
                $document = $this->normalizeDocument($item->cpf ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                if (!$clienteId) {
                    return false;
                }
                return !in_array((int) $clienteId, $propostaClienteIds, true);
            })->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                $leadInfo = $leadByExternal[$externalId] ?? [];
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => $item->prazo,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_consignado'])) {
            $source = 'simulacoes_consignado';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $config['table'];

            try {
                $items = DB::connection(self::EXTERNAL_CONNECTION)->table($table.' as s')
                    ->select([
                        's.id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda_mensal',
                        's.valor_emprestimo',
                        's.origem',
                        's.created_at',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $externalIds = $items->pluck('id')->filter();
            $leadByExternal = $this->loadLocalLeadsByExternal($source, $externalIds);
            $clienteByDocument = $this->loadClientesByDocument($items->pluck('cpf'));
            $clienteIds = collect($clienteByDocument)->values()->unique();
            $propostaClienteIds = $this->loadPropostaClienteIds($clienteIds);

            $data = $data->concat($items->filter(function ($item) use ($clienteByDocument, $propostaClienteIds) {
                $document = $this->normalizeDocument($item->cpf ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                if (!$clienteId) {
                    return false;
                }
                return !in_array((int) $clienteId, $propostaClienteIds, true);
            })->map(function ($item) use ($source, $config, $leadByExternal, $clienteByDocument) {
                $externalId = (int) $item->id;
                $document = $this->normalizeDocument($item->cpf ?? '');
                $clienteId = $document !== '' ? ($clienteByDocument[$document] ?? null) : null;
                $leadInfo = $leadByExternal[$externalId] ?? [];
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $leadInfo['lead_id'] ?? null,
                    'responsavel_id' => $leadInfo['responsavel_id'] ?? null,
                    'cliente_id' => $clienteId ? (int) $clienteId : null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda_mensal,
                    'renda_mensal' => $item->renda_mensal,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $this->normalizeOrigemHost($item->origem ?? '') ?: $item->origem,
                    'origem_url' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $leadInfo['tipo_solicitacao'] ?? $config['tipo_solicitacao'],
                    'validated_at' => $leadInfo['validated_at'] ?? null,
                ];
            }));
        }

        if (!$canViewAll && $userId) {
            $data = $data->filter(function ($item) use ($userId) {
                return (int) ($item['responsavel_id'] ?? 0) === (int) $userId;
            })->values();
        }

        $data = $data->sortByDesc(function ($item) {
            return $item['created_at'] ?? '';
        })->values();

        $data = $data->map(function ($item) {
            $item['telefone'] = $this->normalizeTelefone($item['telefone'] ?? '');
            return $item;
        });

        return response()->json(['data' => $data]);
    }

    public function sendExternalEmail(Request $request, int $externalId)
    {
        $payload = $request->validate([
            'email_to' => ['required', 'string', 'max:500'],
            'external_source' => ['nullable', 'string', 'max:80'],
        ]);

        $source = $this->resolveExternalSourceForId($request, $externalId);
        if ($source === null) {
            return response()->json(['message' => 'Lead externo nao encontrado.'], 404);
        }
        if ($source === 'ambiguous') {
            return response()->json(['message' => 'Fonte externa ambigua. Informe external_source.'], 422);
        }

        $localLead = Lead::query()
            ->where('external_source', $source)
            ->where('external_id', $externalId)
            ->first();
        if ($localLead) {
            $this->authorizeLeadAccess($localLead, $request->user());
        }
        $config = $this->sourceConfig($source);
        $table = $config['table'];
        $external = DB::connection(self::EXTERNAL_CONNECTION)->table($table)->where('id', $externalId)->first();
        if (!$external) {
            return response()->json(['message' => 'Lead externo nao encontrado.'], 404);
        }
        $external->renda = $this->externalRenda($external, $source);
        if (empty($external->cpf)) {
            $external->cpf = $this->resolveExternalDocument($external, $source);
        }
        if ($source === 'simulacoes_consignado' && empty($external->tipo_imovel)) {
            $external->tipo_imovel = 'VEICULO';
        }

        $address = $this->composeExternalAddress($external);
        $mapLink = $this->mapLinkForAddress($address);
        $leadId = $external->id ?? $externalId;
        $subjectName = $this->formatEmailValue($external->nome ?? '');
        $subject = 'LPC Capital - Lead #'.$leadId;
        if ($subjectName !== '-') {
            $subject .= ' - '.$subjectName;
        }
        $html = $this->buildLeadEmailHtml($external, $address, $mapLink);
        $text = $this->buildLeadEmailText($external, $address, $mapLink);

        $fromAddress = config('mail.from.address') ?: env('MAIL_FROM_ADDRESS');
        $fromName = config('mail.from.name') ?: env('MAIL_FROM_NAME');

        $emails = $this->parseEmailList($payload['email_to']);
        if (!$emails) {
            return response()->json(['message' => 'Informe ao menos um email valido.'], 422);
        }
        $invalid = $this->invalidEmails($emails);
        if ($invalid) {
            return response()->json([
                'message' => 'Emails invalidos: '.implode(', ', $invalid),
            ], 422);
        }

        try {
            Mail::send([], [], function ($message) use ($emails, $subject, $fromAddress, $fromName, $html, $text) {
                if ($fromAddress) {
                    $message->from($fromAddress, $fromName ?: null);
                }
                $message->to($emails)
                    ->replyTo('contato@marianalpccapital.com.br', 'LPC Capital')
                    ->subject($subject)
                    ->html($html)
                    ->text($text);
            });
        } catch (\Throwable $exception) {
            logger()->error('Falha ao enviar email do lead externo.', [
                'error' => $exception->getMessage(),
                'external_id' => $externalId,
                'email_to' => $emails,
            ]);
            return response()->json(['message' => 'Falha ao enviar email.'], 500);
        }

        return response()->json(['message' => 'Email enviado com sucesso.']);
    }

    public function importExternal(Request $request, int $externalId)
    {
        $payload = $request->validate([
            'tipo_solicitacao' => ['required', 'string', 'in:'.implode(',', self::TIPO_SOLICITACAO)],
            'external_source' => ['nullable', 'string', 'max:80'],
        ]);

        $source = $this->resolveExternalSourceForId($request, $externalId);
        if ($source === null) {
            return response()->json(['message' => 'Lead externo nao encontrado.'], 404);
        }
        if ($source === 'ambiguous') {
            return response()->json(['message' => 'Fonte externa ambigua. Informe external_source.'], 422);
        }
        $config = $this->sourceConfig($source);
        $table = $config['table'];
        $external = DB::connection(self::EXTERNAL_CONNECTION)->table($table)->where('id', $externalId)->first();
        if (!$external) {
            return response()->json(['message' => 'Lead externo nao encontrado.'], 404);
        }

        $missing = $this->missingExternalFields($external, $source);
        if ($missing) {
            return response()->json([
                'message' => 'Campos obrigatorios ausentes.',
                'fields' => $missing,
            ], 422);
        }

        if (Lead::query()->where('external_source', $source)->where('external_id', $externalId)->exists()) {
            return response()->json(['message' => 'Lead ja importado.'], 409);
        }

        $document = $this->normalizeDocument($this->resolveExternalDocument($external, $source));
        if ($document === '') {
            return response()->json(['message' => 'CPF/CNPJ invalido.'], 422);
        }
        $external->renda = $this->externalRenda($external, $source);
        $external->cpf = $document;

        $lead = DB::transaction(function () use ($external, $payload, $document, $source) {
            $clienteId = $this->resolveClienteId($document, $external);
            $tipoCliente = strlen($document) > 11 ? 'PJ' : 'PF';

            if ($tipoCliente === 'PJ') {
                $pessoaId = $this->ensurePessoaJuridica($clienteId, $external, $document);
                $this->ensureEndereco($clienteId, null, $pessoaId, $external);
            } else {
                $pessoaId = $this->ensurePessoaFisica($clienteId, $external);
                $this->ensureEndereco($clienteId, $pessoaId, null, $external);
            }

            if ($source === 'simulacoes_imobiliarias') {
                $this->ensureImovel($clienteId, $external);
            }

            return Lead::query()->create([
                'nome' => $external->nome,
                'email' => $external->email,
                'telefone' => $external->telefone,
                'origem' => $external->origem ?? 'externo',
                'status' => 'novo',
                'responsavel_id' => auth()->id(),
                'pessoa_fisica_id' => $tipoCliente === 'PF' ? $clienteId : null,
                'pessoa_juridica_id' => $tipoCliente === 'PJ' ? $clienteId : null,
                'external_source' => $source,
                'external_id' => $external->id,
                'validated_at' => now(),
                'tipo_solicitacao' => $payload['tipo_solicitacao'],
            ]);
        });

        $this->dispatchLeadNotification($lead, $lead->responsavel_id);
        return response()->json($lead, 201);
    }

    private function dispatchLeadNotification(Lead $lead, ?int $userId): void
    {
        if (!$userId) {
            return;
        }

        SendLeadAssignedNotifications::dispatch(
            (int) $lead->id,
            (int) $userId,
            now()->toDateTimeString()
        );
    }

    private function canViewAllLeads(?Usuario $user): bool
    {
        if (!$user) {
            return false;
        }
        $role = strtolower(trim((string) ($user->role ?? '')));
        return in_array($role, ['admin', 'administrador', 'gestor', 'gerente', 'superadmin', 'root'], true);
    }

    private function authorizeLeadAccess(Lead $lead, ?Usuario $user): void
    {
        if ($this->canViewAllLeads($user)) {
            return;
        }
        if (!$user || (int) $lead->responsavel_id !== (int) $user->id) {
            abort(403, 'Acesso negado ao lead.');
        }
    }

    private function authorizeLeadAssignment(Lead $lead, ?Usuario $user, int $targetUserId): void
    {
        if ($this->canViewAllLeads($user)) {
            return;
        }
        if (!$user) {
            abort(403, 'Acesso negado.');
        }
        if ((int) $targetUserId !== (int) $user->id) {
            abort(403, 'Atribuicao nao permitida.');
        }
        if ($lead->responsavel_id && (int) $lead->responsavel_id !== (int) $user->id) {
            abort(403, 'Atribuicao nao permitida.');
        }
    }

    private function logLeadAccess(int $leadId, Request $request): void
    {
        LeadAccessLog::query()->create([
            'lead_id' => $leadId,
            'user_id' => $request->user()?->id,
            'action' => 'view',
            'ip_address' => $request->ip(),
            'user_agent' => substr((string) $request->userAgent(), 0, 255),
            'created_at' => now(),
        ]);
    }

    private function resolveLeadSortBy(?string $sortBy): string
    {
        $allowed = ['id', 'created_at', 'status', 'tipo_solicitacao', 'nome'];
        $candidate = strtolower(trim((string) $sortBy));
        if ($candidate === '') {
            return 'id';
        }
        return in_array($candidate, $allowed, true) ? $candidate : 'id';
    }

    private function resolveExternalSortBy(?string $sortBy): ?string
    {
        $allowed = ['created_at', 'tipo_solicitacao', 'nome'];
        $candidate = strtolower(trim((string) $sortBy));
        if ($candidate === '') {
            return null;
        }
        return in_array($candidate, $allowed, true) ? $candidate : null;
    }

    private function resolveSortDir(?string $sortDir): string
    {
        return strtolower((string) $sortDir) === 'asc' ? 'asc' : 'desc';
    }

    private function resolvePerPage(?string $value): int
    {
        $perPage = (int) $value;
        if ($perPage <= 0) {
            return 20;
        }
        return min($perPage, 200);
    }

    private function sourceConfig(string $source): array
    {
        return self::EXTERNAL_SOURCES[$source] ?? self::EXTERNAL_SOURCES['simulacoes_imobiliarias'];
    }

    private function resolveExternalDocument(object $external, string $source): string
    {
        if ($source === 'simulacoes_auto_equity_pj') {
            return (string) ($external->empresa_cnpj ?? $external->cpf ?? '');
        }
        if ($source === 'simulacoes_financiamento_veiculo') {
            if (($external->tipo_pessoa ?? '') === 'PJ') {
                return (string) ($external->cnpj ?? '');
            }
            return (string) ($external->cpf ?? '');
        }
        if ($source === 'simulacoes_credito_condominio') {
            return (string) ($external->condominio_cnpj ?? '');
        }
        if ($source === 'simulacoes_credito_aval_digital_fgi') {
            return (string) ($external->cnpj ?? '');
        }
        if ($source === 'simulacoes_creditos_pj_diversos') {
            return (string) ($external->cnpj ?? '');
        }
        return (string) ($external->cpf ?? '');
    }

    private function resolveExternalSourceForId(Request $request, int $externalId): ?string
    {
        $explicit = $request->input('external_source') ?? $request->query('external_source');
        if (is_string($explicit) && $explicit !== '') {
            return isset(self::EXTERNAL_SOURCES[$explicit]) ? $explicit : null;
        }

        $found = [];
        foreach (self::EXTERNAL_SOURCES as $key => $config) {
            try {
                $exists = DB::connection(self::EXTERNAL_CONNECTION)
                    ->table($config['table'])
                    ->where('id', $externalId)
                    ->exists();
            } catch (\Throwable $exception) {
                continue;
            }
            if ($exists) {
                $found[] = $key;
            }
        }

        if (!$found) {
            return null;
        }
        if (count($found) > 1) {
            return 'ambiguous';
        }

        return $found[0];
    }

    private function externalRenda(object $external, string $source): float
    {
        if ($source === 'simulacoes_consignado') {
            return (float) ($external->renda_mensal ?? 0);
        }
        if ($source === 'simulacoes_auto_equity_pj') {
            return (float) ($external->faturamento_12m ?? 0);
        }
        if ($source === 'simulacoes_financiamento_veiculo') {
            $renda = (float) ($external->renda ?? 0);
            if ($renda <= 0 && ($external->tipo_pessoa ?? '') === 'PJ') {
                return (float) ($external->faturamento_anual ?? 0);
            }
            return $renda;
        }
        if ($source === 'simulacoes_credito_condominio') {
            return (float) ($external->arrecadacao_mensal ?? 0);
        }
        if ($source === 'simulacoes_credito_aval_digital_fgi') {
            return (float) ($external->faturamento_anual ?? 0);
        }
        if ($source === 'simulacoes_creditos_pj_diversos') {
            return (float) ($external->faturamento_anual ?? 0);
        }
        if ($source === 'simulacoes_credito_construcao') {
            return 0;
        }
        return (float) ($external->renda ?? 0);
    }

    private function missingExternalFields(object $external, string $source): array
    {
        if ($source === 'simulacoes_consignado') {
            $required = [
                'nome',
                'email',
                'telefone',
                'cpf',
                'renda_mensal',
                'cep',
                'endereco',
                'numero',
                'bairro',
                'cidade',
                'estado',
                'valor_emprestimo',
                'data_nascimento',
            ];
        } elseif ($source === 'simulacoes_auto_equity_pf') {
            $required = [
                'nome',
                'email',
                'telefone',
                'cpf',
                'renda',
                'data_nascimento',
                'cep',
                'endereco',
                'numero',
                'bairro',
                'cidade',
                'estado',
                'valor_emprestimo',
                'veiculo_tipo',
                'veiculo_marca',
                'veiculo_modelo',
                'ano_fabricacao',
                'ano_modelo',
                'placa',
                'veiculo_nome',
                'alienacao',
            ];
        } elseif ($source === 'simulacoes_auto_equity_pj') {
            $required = [
                'empresa_nome',
                'empresa_cnpj',
                'email',
                'telefone',
                'faturamento_12m',
                'valor_emprestimo',
                'cep',
                'endereco',
                'numero',
                'bairro',
                'cidade',
                'estado',
                'socio_nome',
                'socio_cpf',
                'socio_data_nascimento',
                'socio_renda',
                'veiculo_marca',
                'veiculo_modelo',
                'ano_fabricacao',
                'ano_modelo',
                'placa',
                'veiculo_nome_empresa',
            ];
        } elseif ($source === 'simulacoes_financiamento_veiculo') {
            $required = [
                'tipo_pessoa',
                'nome',
                'email',
                'telefone',
                'condicao_veiculo',
                'tipo_veiculo',
                'marca',
                'modelo',
                'ano_fabricacao',
                'ano_modelo',
                'valor_veiculo',
            ];
            if (($external->tipo_pessoa ?? '') === 'PJ') {
                $required[] = 'cnpj';
            } else {
                $required[] = 'cpf';
            }
        } elseif ($source === 'simulacoes_consorcio') {
            $required = [
                'nome',
                'email',
                'telefone',
                'cpf',
                'renda',
                'tipo_bem',
                'valor_bem',
                'lance_embutido',
                'lance_proprio',
                'valor_parcela_max',
                'prazo_desejado',
            ];
        } elseif ($source === 'simulacoes_financiamento_imobiliario_pf') {
            $required = [
                'nome',
                'email',
                'telefone',
                'cpf',
                'data_nascimento',
                'renda',
                'tipo_imovel',
                'uf_imovel',
                'valor_imovel',
                'valor_financiamento',
                'prazo',
                'imovel_definido',
                'usar_fgts',
                'incluir_itbi',
            ];
        } elseif ($source === 'simulacoes_credito_construcao') {
            $required = [
                'nome',
                'email',
                'telefone',
                'cpf',
                'valor_proposta',
                'valor_imovel_acabado',
                'tipo_construcao',
                'matricula_iptu',
                'percentual_construcao',
                'alvara',
                'art_rrt',
                'projeto_arquitetonico',
                'cep',
                'endereco',
                'numero',
                'bairro',
                'cidade',
                'estado',
            ];
        } elseif ($source === 'simulacoes_credito_condominio') {
            $required = [
                'condominio_nome',
                'condominio_cnpj',
                'email',
                'telefone',
                'cep',
                'endereco',
                'cidade',
                'estado',
                'valor_imovel',
                'arrecadacao_mensal',
                'valor_credito',
                'carencia',
                'finalidade',
                'forma_custeio_credito',
                'admin_nome',
                'admin_cnpj',
                'admin_telefone',
                'admin_email',
                'sindico_nome',
                'sindico_profissional',
                'sindico_cpf',
                'sindico_email',
                'sindico_celular',
            ];
        } elseif ($source === 'simulacoes_credito_aval_digital_fgi') {
            $required = [
                'razao_social',
                'cnpj',
                'email',
                'telefone',
                'faturamento_anual',
                'valor_solicitado',
                'prazo_meses',
                'segmento',
                'funcionarios',
            ];
        } elseif ($source === 'simulacoes_creditos_pj_diversos') {
            $required = [
                'razao_social',
                'cnpj',
                'email',
                'telefone',
                'tipo_operacao',
                'faturamento_anual',
                'valor_proposta',
            ];
        } else {
            $required = [
                'nome',
                'email',
                'telefone',
                'cpf',
                'renda',
                'cep',
                'endereco',
                'numero',
                'bairro',
                'cidade',
                'estado',
                'valor_imovel',
                'tipo_imovel',
                'propriedade',
                'metragem',
                'financiado',
                'condominio',
                'valor_emprestimo',
                'prazo',
            ];
        }

        $missing = [];
        foreach ($required as $field) {
            if (!isset($external->$field) || $external->$field === '' || $external->$field === null) {
                $missing[] = $field;
            }
        }

        return $missing;
    }

    private function normalizeDocument(?string $value): string
    {
        return preg_replace('/\D+/', '', (string) $value);
    }

    private function normalizeTelefone(?string $value): string
    {
        return preg_replace('/\D+/', '', (string) $value) ?: '';
    }

    private function normalizeOrigemHost(?string $value): string
    {
        $raw = trim((string) $value);
        if ($raw === '') {
            return '';
        }
        $host = parse_url($raw, PHP_URL_HOST);
        if ($host) {
            return strtolower($host);
        }
        if (str_contains($raw, '/')) {
            $parts = explode('/', $raw);
            $candidate = $parts[0] ?? '';
            return strtolower($candidate);
        }
        return strtolower($raw);
    }

    private function loadLocalLeadsByExternal(string $source, $externalIds): array
    {
        if ($externalIds->isEmpty()) {
            return [];
        }

        $rows = Lead::query()
            ->select(['id', 'external_id', 'external_source', 'tipo_solicitacao', 'validated_at', 'responsavel_id'])
            ->where('external_source', $source)
            ->whereIn('external_id', $externalIds->values()->all())
            ->get();

        $map = [];
        foreach ($rows as $row) {
            $externalId = (int) $row->external_id;
            $map[$externalId] = [
                'lead_id' => (int) $row->id,
                'tipo_solicitacao' => $row->tipo_solicitacao,
                'validated_at' => $row->validated_at,
                'responsavel_id' => $row->responsavel_id ? (int) $row->responsavel_id : null,
            ];
        }

        return $map;
    }

    private function loadClientesByDocument($documents): array
    {
        if ($documents->isEmpty() || !Schema::hasTable('clientes')) {
            return [];
        }

        $normalized = $documents->map(function ($value) {
            return $this->normalizeDocument($value);
        })->filter()->unique()->values();

        if ($normalized->isEmpty()) {
            return [];
        }

        $rows = DB::table('clientes')
            ->select(['id', 'cpf_cnpj'])
            ->whereIn(DB::raw("REPLACE(REPLACE(REPLACE(REPLACE(cpf_cnpj, '.', ''), '-', ''), '/', ''), ' ', '')"), $normalized->all())
            ->get();

        $map = [];
        foreach ($rows as $row) {
            $document = $this->normalizeDocument($row->cpf_cnpj ?? '');
            if ($document !== '') {
                $map[$document] = (int) $row->id;
            }
        }

        return $map;
    }

    private function loadPropostaClienteIds($clienteIds): array
    {
        if ($clienteIds->isEmpty() || !Schema::hasTable('propostas_credito')) {
            return [];
        }

        return DB::table('propostas_credito')
            ->whereIn('cliente_id', $clienteIds->values()->all())
            ->pluck('cliente_id')
            ->map(fn ($id) => (int) $id)
            ->values()
            ->all();
    }

    private function resolveClienteId(string $document, object $external): int
    {
        $cliente = Cliente::query()->where('cpf_cnpj', $document)->first();
        if ($cliente) {
            return (int) $cliente->id;
        }

        $tipoCliente = strlen($document) > 11 ? 'PJ' : 'PF';

        $cliente = Cliente::query()->create([
            'tipo_cliente' => $tipoCliente,
            'nome_razao' => $external->nome,
            'cpf_cnpj' => $document,
            'renda' => $external->renda,
            'usuario_responsavel' => auth()->id(),
            'status' => 'lead',
        ]);

        return (int) $cliente->id;
    }

    private function ensurePessoaFisica(int $clienteId, object $external): int
    {
        $pessoa = PessoaFisica::query()->where('cliente_id', $clienteId)->first();
        if ($pessoa) {
            return (int) $pessoa->cliente_id;
        }

        $pessoa = PessoaFisica::query()->create([
            'cliente_id' => $clienteId,
            'renda_mensal' => $external->renda,
        ]);

        return (int) $pessoa->cliente_id;
    }

    private function ensurePessoaJuridica(int $clienteId, object $external, string $document): int
    {
        $pessoa = PessoaJuridica::query()->where('cliente_id', $clienteId)->first();
        if ($pessoa) {
            return (int) $pessoa->cliente_id;
        }

        $pessoa = PessoaJuridica::query()->create([
            'cliente_id' => $clienteId,
            'cnpj' => $document,
            'renda' => $external->renda,
        ]);

        return (int) $pessoa->cliente_id;
    }

    private function ensureEndereco(int $clienteId, ?int $pessoaFisicaId, ?int $pessoaJuridicaId, object $external): void
    {
        if (!Schema::hasTable('enderecos')) {
            return;
        }

        $data = $this->filterColumns('enderecos', [
            'cliente_id' => $clienteId,
            'logradouro' => $external->endereco ?? null,
            'numero' => $external->numero ?? null,
            'bairro' => $external->bairro ?? null,
            'cidade' => $external->cidade ?? null,
            'uf' => $external->estado ?? null,
            'cep' => $external->cep ?? null,
            'pessoa_fisica_id' => $pessoaFisicaId,
            'pessoa_juridica_id' => $pessoaJuridicaId,
        ], true);

        if (!$data) {
            return;
        }

        $match = [];
        if (isset($data['cliente_id'])) {
            $match['cliente_id'] = $data['cliente_id'];
        }
        if (isset($data['pessoa_fisica_id'])) {
            $match['pessoa_fisica_id'] = $data['pessoa_fisica_id'];
        }
        if (isset($data['pessoa_juridica_id'])) {
            $match['pessoa_juridica_id'] = $data['pessoa_juridica_id'];
        }
        if (isset($data['cep'])) {
            $match['cep'] = $data['cep'];
        }
        if (isset($data['numero'])) {
            $match['numero'] = $data['numero'];
        }

        if ($match) {
            DB::table('enderecos')->updateOrInsert($match, $data);
            return;
        }

        DB::table('enderecos')->insert($data);
    }

    private function ensureImovel(int $clienteId, object $external): ?int
    {
        if (!Schema::hasTable('imoveis')) {
            return null;
        }

        $tipoImovelId = $this->resolveTipoImovelId($external->tipo_imovel ?? null);

        $data = $this->filterColumns('imoveis', [
            'cliente_id' => $clienteId,
            'tipo_imovel_id' => $tipoImovelId,
            'cep' => $external->cep ?? null,
            'logradouro' => $external->endereco ?? null,
            'numero' => $external->numero ?? null,
            'bairro' => $external->bairro ?? null,
            'cidade' => $external->cidade ?? null,
            'estado' => $external->estado ?? null,
            'metragem_m2' => $external->metragem ?? null,
            'valor_avaliacao' => $external->valor_imovel ?? null,
            'valor_credito' => $external->valor_emprestimo ?? null,
            'saldo_devedor' => $external->saldo_financiado ?? null,
            'em_condominio' => $this->parseYesNo($external->condominio ?? null),
            'propriedade' => $external->propriedade ?? null,
        ], true);

        if (!$data) {
            return null;
        }

        $match = [];
        if (isset($data['cliente_id'])) {
            $match['cliente_id'] = $data['cliente_id'];
        }
        if (isset($data['cep'])) {
            $match['cep'] = $data['cep'];
        }
        if (isset($data['numero'])) {
            $match['numero'] = $data['numero'];
        }

        if ($match) {
            $existing = DB::table('imoveis')->where($match)->first();
            if ($existing && isset($existing->id)) {
                return (int) $existing->id;
            }
        }

        return (int) DB::table('imoveis')->insertGetId($data);
    }

    private function parseYesNo(?string $value): ?bool
    {
        $normalized = strtolower(trim((string) $value));
        if ($normalized === '') {
            return null;
        }
        if (in_array($normalized, ['sim', 's', 'yes', 'y', '1'], true)) {
            return true;
        }
        if (in_array($normalized, ['nao', 'n', 'no', '0'], true)) {
            return false;
        }

        return null;
    }

    private function formatEmailValue($value): string
    {
        if ($value === null) {
            return '-';
        }
        $text = trim((string) $value);
        return $text === '' ? '-' : $text;
    }

    private function formatEmailMoney($value): string
    {
        if ($value === null || $value === '') {
            return '-';
        }
        if (!is_numeric($value)) {
            return $this->formatEmailValue($value);
        }
        return 'R$ '.number_format((float) $value, 2, ',', '.');
    }

    private function composeExternalAddress(object $external): string
    {
        $parts = [
            $external->endereco ?? null,
            $external->numero ?? null,
            $external->bairro ?? null,
            $external->cidade ?? null,
            $external->estado ?? null,
            $external->cep ?? null,
        ];

        $clean = array_values(array_filter(array_map(function ($value) {
            $value = trim((string) $value);
            return $value === '' ? null : $value;
        }, $parts)));

        return implode(', ', $clean);
    }

    private function mapLinkForAddress(string $address): string
    {
        if ($address === '') {
            return '';
        }
        return 'https://www.google.com/maps/search/?api=1&query='.urlencode($address);
    }

    private function publicLogoUrl(): string
    {
        return 'https://marianalpccapital.com.br/img/Logo_LPC_Capital.png';
    }

    private function buildLeadEmailHtml(object $external, string $address, string $mapLink): string
    {
        $addressValue = $address !== '' ? $address : '-';
        $mapValue = $mapLink !== '' ? $mapLink : '-';
        $title = $this->formatEmailValue($external->nome ?? 'Lead');
        $logoUrl = $this->publicLogoUrl();

        $sections = [
            'Cliente' => [
                'ID' => $this->formatEmailValue($external->id ?? ''),
                'Nome' => $this->formatEmailValue($external->nome ?? ''),
                'Email' => $this->formatEmailValue($external->email ?? ''),
                'Telefone' => $this->formatEmailValue($external->telefone ?? ''),
                'CPF' => $this->formatEmailValue($external->cpf ?? ''),
                'Renda' => $this->formatEmailMoney($external->renda ?? null),
                'Origem' => $this->formatEmailValue($external->origem ?? ''),
            ],
            'Endereco' => [
                'Endereco' => $addressValue,
                'Complemento' => $this->formatEmailValue($external->complemento ?? ''),
                'Cidade/UF' => $this->formatEmailValue($external->cidade ?? '').' / '.$this->formatEmailValue($external->estado ?? ''),
                'Mapa' => $mapValue,
            ],
            'Imovel' => [
                'Tipo' => $this->formatEmailValue($external->tipo_imovel ?? ''),
                'Propriedade' => $this->formatEmailValue($external->propriedade ?? ''),
                'Metragem' => $this->formatEmailValue($external->metragem ?? ''),
                'Valor' => $this->formatEmailMoney($external->valor_imovel ?? null),
                'Financiado' => $this->formatEmailValue($external->financiado ?? ''),
                'Saldo' => $this->formatEmailMoney($external->saldo_financiado ?? null),
                'Condominio' => $this->formatEmailValue($external->condominio ?? ''),
            ],
            'Emprestimo' => [
                'Valor solicitado' => $this->formatEmailMoney($external->valor_emprestimo ?? null),
                'Prazo' => isset($external->prazo) ? $this->formatEmailValue($external->prazo).' meses' : '-',
                'Tipo solicitacao' => $this->formatEmailValue($external->tipo_solicitacao ?? ''),
            ],
        ];

        $html = '<!doctype html><html><head><meta charset="utf-8">';
        $html .= '<style>';
        $html .= 'body{font-family:Arial,sans-serif;color:#0f172a;margin:24px;}';
        $html .= 'h1{font-size:22px;margin:0;}';
        $html .= 'h2{font-size:13px;margin:18px 0 8px;text-transform:uppercase;letter-spacing:0.12em;color:#475569;}';
        $html .= 'table{width:100%;border-collapse:collapse;}';
        $html .= 'th,td{text-align:left;padding:6px 8px;border-bottom:1px solid #e2e8f0;font-size:12px;}';
        $html .= 'th{width:32%;color:#475569;font-weight:600;}';
        $html .= '.subtle{color:#64748b;font-size:12px;}';
        $html .= '.brand{display:flex;align-items:center;gap:16px;margin-bottom:18px;}';
        $html .= '.brand img{height:36px;width:auto;}';
        $html .= '</style></head><body>';
        $html .= '<div class="brand">';
        if ($logoUrl !== '') {
            $html .= '<img src="'.e($logoUrl).'" alt="LPC Capital" />';
        }
        $html .= '<div><h1>Lead '.e($title).'</h1>';
        $html .= '<div class="subtle">Gerado em '.e(now()->format('d/m/Y H:i')).'</div></div></div>';

        foreach ($sections as $sectionTitle => $rows) {
            $html .= '<section><h2>'.e($sectionTitle).'</h2><table>';
            foreach ($rows as $label => $value) {
                $html .= '<tr><th>'.e($label).'</th><td>'.e($value).'</td></tr>';
            }
            $html .= '</table></section>';
        }

        $html .= '</body></html>';

        return $html;
    }

    private function buildLeadEmailText(object $external, string $address, string $mapLink): string
    {
        $addressValue = $address !== '' ? $address : '-';
        $mapValue = $mapLink !== '' ? $mapLink : '-';

        $lines = [
            'Lead '.($this->formatEmailValue($external->nome ?? '')),
            'Gerado em '.now()->format('d/m/Y H:i'),
            '',
            'Cliente',
            'ID: '.$this->formatEmailValue($external->id ?? ''),
            'Nome: '.$this->formatEmailValue($external->nome ?? ''),
            'Email: '.$this->formatEmailValue($external->email ?? ''),
            'Telefone: '.$this->formatEmailValue($external->telefone ?? ''),
            'CPF: '.$this->formatEmailValue($external->cpf ?? ''),
            'Renda: '.$this->formatEmailMoney($external->renda ?? null),
            'Origem: '.$this->formatEmailValue($external->origem ?? ''),
            '',
            'Endereco',
            'Endereco: '.$addressValue,
            'Complemento: '.$this->formatEmailValue($external->complemento ?? ''),
            'Cidade/UF: '.$this->formatEmailValue($external->cidade ?? '').' / '.$this->formatEmailValue($external->estado ?? ''),
            'Mapa: '.$mapValue,
            '',
            'Imovel',
            'Tipo: '.$this->formatEmailValue($external->tipo_imovel ?? ''),
            'Propriedade: '.$this->formatEmailValue($external->propriedade ?? ''),
            'Metragem: '.$this->formatEmailValue($external->metragem ?? ''),
            'Valor: '.$this->formatEmailMoney($external->valor_imovel ?? null),
            'Financiado: '.$this->formatEmailValue($external->financiado ?? ''),
            'Saldo: '.$this->formatEmailMoney($external->saldo_financiado ?? null),
            'Condominio: '.$this->formatEmailValue($external->condominio ?? ''),
            '',
            'Emprestimo',
            'Valor solicitado: '.$this->formatEmailMoney($external->valor_emprestimo ?? null),
            'Prazo: '.(isset($external->prazo) ? $this->formatEmailValue($external->prazo).' meses' : '-'),
            'Tipo solicitacao: '.$this->formatEmailValue($external->tipo_solicitacao ?? ''),
        ];

        return implode("\n", $lines);
    }

    private function parseEmailList(string $value): array
    {
        $parts = preg_split('/[\s,;]+/', $value);
        $emails = [];
        foreach ($parts as $part) {
            $email = trim((string) $part);
            if ($email === '') {
                continue;
            }
            $email = trim($email, " \t\n\r\0\x0B<>;,");
            $email = rtrim($email, '.');
            if ($email !== '') {
                $emails[] = $email;
            }
        }
        return array_values(array_unique($emails));
    }

    private function invalidEmails(array $emails): array
    {
        $invalid = [];
        foreach ($emails as $email) {
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $invalid[] = $email;
            }
        }
        return $invalid;
    }

    private function resolveTipoImovelId(?string $tipo): ?int
    {
        $normalized = strtolower(trim((string) $tipo));
        if ($normalized === '') {
            return null;
        }

        if (Schema::hasTable('tipo_imoveis')) {
            $columns = Schema::getColumnListing('tipo_imoveis');
            $nameColumn = null;
            foreach (['nome', 'descricao', 'titulo'] as $candidate) {
                if (in_array($candidate, $columns, true)) {
                    $nameColumn = $candidate;
                    break;
                }
            }
            if ($nameColumn) {
                $row = DB::table('tipo_imoveis')
                    ->whereRaw('LOWER('.$nameColumn.') = ?', [$normalized])
                    ->first();
                if ($row && isset($row->id)) {
                    return (int) $row->id;
                }
            }
        }

        $map = [
            'apartamento' => 1,
            'casa' => 2,
            'sala comercial' => 3,
            'terreno' => 4,
            'outros' => 5,
        ];

        return $map[$normalized] ?? null;
    }

    private function filterColumns(string $table, array $data, bool $dropEmpty = false): array
    {
        $columns = Schema::getColumnListing($table);
        $filtered = [];
        foreach ($data as $key => $value) {
            if (!in_array($key, $columns, true)) {
                continue;
            }
            if ($dropEmpty && ($value === null || $value === '')) {
                continue;
            }
            $filtered[$key] = $value;
        }

        return $filtered;
    }

    private function ensureExternalDocumentsTable(): void
    {
        $table = self::EXTERNAL_DOCUMENTS_TABLE;
        $sql = <<<'SQL'
CREATE TABLE IF NOT EXISTS %s (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    simulacao_id BIGINT UNSIGNED NOT NULL,
    arquivo VARCHAR(255) NOT NULL,
    tipo VARCHAR(60) NULL,
    created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    INDEX simulacoes_documentos_simulacao_id_index (simulacao_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
SQL;

        DB::connection(self::EXTERNAL_CONNECTION)->statement(sprintf($sql, $table));
    }

    private function loadExternalDocuments($externalIds): array
    {
        if ($externalIds->isEmpty()) {
            return [];
        }

        try {
            $rows = DB::connection(self::EXTERNAL_CONNECTION)->table(self::EXTERNAL_DOCUMENTS_TABLE)
                ->whereIn('simulacao_id', $externalIds->values()->all())
                ->orderBy('id')
                ->get();
        } catch (\Throwable $exception) {
            return [];
        }

        $documents = [];
        foreach ($rows as $row) {
            $documents[$row->simulacao_id][] = [
                'id' => $row->id,
                'arquivo' => $row->arquivo,
                'tipo' => $row->tipo,
            ];
        }

        return $documents;
    }
}







