<?php

namespace App\Http\Controllers;

use App\Models\Parceiro;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;

class ParceiroController extends Controller
{
    public function index()
    {
        $rows = Parceiro::query()->orderBy('nome')->get();
        return response()->json(['data' => $rows]);
    }

    public function show(Parceiro $parceiro)
    {
        return response()->json($parceiro);
    }

    public function store(Request $request)
    {
        $data = $this->validatePayload($request, true);
        $data = $this->encryptSecrets($data);
        $parceiro = Parceiro::query()->create($data);
        return response()->json($parceiro, 201);
    }

    public function update(Request $request, Parceiro $parceiro)
    {
        $data = $this->validatePayload($request, false);
        $data = $this->encryptSecrets($data);
        $parceiro->fill($data)->save();
        return response()->json($parceiro);
    }

    public function destroy(Parceiro $parceiro)
    {
        $parceiro->delete();
        return response()->noContent();
    }

    private function validatePayload(Request $request, bool $creating): array
    {
        $rules = [
            'nome' => ['required', 'string', 'max:160'],
            'tipo' => ['nullable', 'string', 'max:20'],
            'cnpj' => ['nullable', 'string', 'max:30'],
            'canal_envio' => ['nullable', 'string', 'in:api,email'],
            'tipo_integracao' => ['nullable', 'string', 'max:60'],
            'contato_nome' => ['nullable', 'string', 'max:120'],
            'contato_email' => ['nullable', 'email', 'max:160'],
            'contato_telefone' => ['nullable', 'string', 'max:40'],
            'api_url' => ['nullable', 'string', 'max:255'],
            'api_base_url' => ['nullable', 'string', 'max:255'],
            'api_auth_url' => ['nullable', 'string', 'max:255'],
            'api_client_id' => ['nullable', 'string', 'max:255'],
            'api_client_secret' => ['nullable', 'string', 'max:255'],
            'api_scope' => ['nullable', 'string', 'max:255'],
            'api_audience' => ['nullable', 'string', 'max:255'],
            'api_token' => ['nullable', 'string', 'max:500'],
            'api_token_expires_at' => ['nullable', 'date'],
            'api_key' => ['nullable', 'string', 'max:255'],
            'api_headers' => ['nullable', 'string'],
            'api_payload_map' => ['nullable', 'string'],
            'api_enabled' => ['nullable', 'boolean'],
            'ativo' => ['nullable', 'boolean'],
        ];

        if (!$creating) {
            foreach ($rules as $key => $rule) {
                $rules[$key] = array_merge(['sometimes'], (array) $rule);
            }
        }

        return $request->validate($rules);
    }

    private function encryptSecrets(array $data): array
    {
        foreach (['api_client_secret', 'api_key', 'api_token'] as $field) {
            if (array_key_exists($field, $data) && $data[$field]) {
                $data[$field] = Crypt::encryptString($data[$field]);
            }
        }
        return $data;
    }
}
