<?php

namespace App\Http\Controllers;

use App\Models\Cliente;
use App\Models\ContratoDocumentoSolicitacao;
use App\Models\Lead;
use App\Models\Parceiro;
use App\Models\PropostaCredito;
use App\Models\PropostaDocumento;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Crypt;
use Carbon\Carbon;

class PropostaCreditoController extends Controller
{
    private const EXTERNAL_CONNECTION = 'external_mysql';

    public function index()
    {
        $propostas = PropostaCredito::query()
            ->with('cliente')
            ->orderByDesc('id')
            ->paginate(20);

        return response()->json($propostas);
    }

    public function create()
    {
        return response()->json(['message' => 'create']);
    }

    public function store(Request $request)
    {
        $registro = PropostaCredito::query()->create($request->only([
            'cliente_id',
            'valor_solicitado',
            'prazo_meses',
            'finalidade',
            'status_proposta_id',
        ]));

        return response()->json($registro->load('cliente'), 201);
    }

    public function show(int $id)
    {
        $proposta = PropostaCredito::query()
            ->with(['cliente', 'propostaDocumentos', 'solicitacoesDocumentos'])
            ->findOrFail($id);

        $payload = $proposta->toArray();
        $payload['documentos'] = $this->mapDocumentos($proposta->propostaDocumentos);
        $payload['solicitacoes_documentos'] = $this->mapSolicitacoes($proposta->solicitacoesDocumentos, 'proposta');
        $contratoIds = $proposta->contratoCreditos()->pluck('id');
        $contratoSolicitacoes = $contratoIds->isNotEmpty()
            ? ContratoDocumentoSolicitacao::query()
                ->whereIn('contrato_id', $contratoIds)
                ->orderByDesc('id')
                ->get()
            : collect();
        $payload['solicitacoes_documentos_contrato'] = $this->mapSolicitacoes($contratoSolicitacoes, 'contrato');
        $payload['endereco'] = $this->loadEndereco($proposta->cliente_id);
        $payload['contato'] = $this->loadContato($proposta->cliente_id);

        return response()->json($payload);
    }

    public function statusOptions()
    {
        try {
            if (!Schema::hasTable('status_proposta')) {
                return response()->json([]);
            }
        } catch (\Throwable $exception) {
            return response()->json([]);
        }

        $rows = DB::table('status_proposta')
            ->select(['id', 'nome'])
            ->orderBy('id')
            ->get();

        return response()->json($rows);
    }

    public function etapasResumo()
    {
        $resumo = [
            'documentacao' => DB::table('propostas_credito')->whereNotNull('documentacao_at')->count(),
            'vistoria' => DB::table('propostas_credito')->whereNotNull('vistoria_at')->count(),
            'avaliacao' => DB::table('propostas_credito')->whereNotNull('avaliacao_at')->count(),
            'comite' => DB::table('propostas_credito')->whereNotNull('comite_at')->count(),
        ];

        return response()->json($resumo);
    }

    public function updateEtapa(Request $request, PropostaCredito $proposta)
    {
        $data = $request->validate([
            'etapa' => ['required', 'string', 'in:documentacao,vistoria,avaliacao,comite'],
            'checked' => ['required', 'boolean'],
        ]);

        $column = $data['etapa'].'_at';
        $value = $data['checked'] ? now() : null;

        $proposta->forceFill([$column => $value])->save();

        return response()->json([
            'etapa' => $data['etapa'],
            'value' => $value ? $value->format('Y-m-d H:i:s') : null,
        ]);
    }

    public function edit(int $id)
    {
        return response()->json(PropostaCredito::query()->findOrFail($id));
    }

    public function update(Request $request, int $id)
    {
        $registro = PropostaCredito::query()->findOrFail($id);
        $registro->update($request->only([
            'valor_solicitado',
            'prazo_meses',
            'finalidade',
            'status_proposta_id',
        ]));

        return response()->json($registro->load('cliente'));
    }

    public function destroy(int $id)
    {
        PropostaCredito::query()->whereKey($id)->delete();
        return response()->noContent();
    }

    public function suggest(int $clienteId)
    {
        $cliente = Cliente::query()->find($clienteId);
        if (!$cliente) {
            return response()->json(['message' => 'Cliente nao encontrado.'], 404);
        }

        $defaultStatusId = $this->resolveDefaultStatusId();

        $proposta = PropostaCredito::query()
            ->where('cliente_id', $clienteId)
            ->orderByDesc('id')
            ->first();

        if ($proposta) {
            return response()->json([
                'source' => 'proposta',
                'valor_solicitado' => $proposta->valor_solicitado,
                'prazo_meses' => $proposta->prazo_meses,
                'finalidade' => $proposta->finalidade,
                'status_proposta_id' => $proposta->status_proposta_id ?? $defaultStatusId,
            ]);
        }

        $lead = Lead::query()
            ->where(function ($query) use ($clienteId) {
                $query->where('pessoa_fisica_id', $clienteId)
                    ->orWhere('pessoa_juridica_id', $clienteId);
            })
            ->orderByDesc('id')
            ->first();

        $external = $this->loadExternalSuggestion($cliente, $lead);
        if ($external) {
            return response()->json([
                'source' => 'lead',
                'valor_solicitado' => $external->valor_emprestimo ?? null,
                'prazo_meses' => $external->prazo ?? null,
                'finalidade' => $lead?->tipo_solicitacao ?? null,
                'status_proposta_id' => $defaultStatusId,
            ]);
        }

        return response()->json([
            'source' => null,
            'valor_solicitado' => null,
            'prazo_meses' => null,
            'finalidade' => null,
            'status_proposta_id' => $defaultStatusId,
        ]);
    }

    public function enviarParceiro(Request $request, PropostaCredito $proposta)
    {
        $payload = $request->validate([
            'parceiro_id' => ['required', 'integer', 'exists:parceiros,id'],
            'canal' => ['nullable', 'string', 'in:api,email'],
        ]);

        $parceiro = Parceiro::query()->findOrFail($payload['parceiro_id']);
        if (!$parceiro->ativo) {
            return response()->json(['message' => 'Parceiro inativo.'], 422);
        }

        $canal = $payload['canal'] ?: ($parceiro->canal_envio ?: 'email');
        $documentosRows = PropostaDocumento::query()
            ->where('proposta_id', $proposta->id)
            ->orderBy('id')
            ->get();
        $tipoCliente = $proposta->cliente?->tipo_cliente ?? null;
        $requiredDocs = $this->documentosObrigatorios($tipoCliente);
        if (!$requiredDocs) {
            return response()->json(['message' => 'Tipo de cliente nao definido para validar documentos.'], 422);
        }
        [$missingDocs, $pendingDocs] = $this->validateDocumentosObrigatorios($requiredDocs, $documentosRows);
        if ($missingDocs) {
            return response()->json([
                'message' => 'Faltam documentos obrigatorios: '.implode(', ', $missingDocs).'.',
            ], 422);
        }
        if ($pendingDocs) {
            return response()->json([
                'message' => 'Documentos pendentes de validacao: '.implode(', ', $pendingDocs).'.',
            ], 422);
        }

        $data = $this->buildParceiroPayload($proposta, $documentosRows);
        $documentos = $data['documentos'] ?? [];
        if (!$documentos) {
            return response()->json(['message' => 'Proposta sem documentos. Anexe antes de enviar ao parceiro.'], 422);
        }

        if ($canal === 'api') {
            if (!$parceiro->api_url) {
                return response()->json(['message' => 'API do parceiro nao configurada.'], 422);
            }
            $result = $this->sendParceiroApi($parceiro, $data);
            if (!$result['ok']) {
                return response()->json([
                    'message' => 'Falha ao enviar para a API do parceiro.',
                    'status' => $result['status'],
                ], 502);
            }
            return response()->json([
                'message' => 'Proposta enviada para a API do parceiro.',
                'status' => $result['status'],
            ]);
        }

        if (!$parceiro->contato_email) {
            return response()->json(['message' => 'Email do parceiro nao configurado.'], 422);
        }

        $this->sendParceiroEmail($parceiro, $data);

        return response()->json(['message' => 'Proposta enviada por email ao parceiro.']);
    }

    private function loadExternalSuggestion(Cliente $cliente, ?Lead $lead): ?object
    {
        $query = DB::connection(self::EXTERNAL_CONNECTION)->table('simulacoes_imobiliarias');

        if ($lead && $lead->external_id) {
            return $query
                ->select(['id', 'valor_emprestimo', 'prazo'])
                ->where('id', $lead->external_id)
                ->first();
        }

        $document = $this->normalizeDocument($cliente->cpf_cnpj ?? '');
        if ($document === '') {
            return null;
        }

        return $query->from('simulacoes_imobiliarias as s')
            ->select(['s.id', 's.valor_emprestimo', 's.prazo'])
            ->whereRaw(
                "REPLACE(REPLACE(REPLACE(REPLACE(s.cpf, '.', ''), '-', ''), '/', ''), ' ', '') = ?",
                [$document]
            )
            ->orderByDesc('s.id')
            ->first();
    }

    private function normalizeDocument(string $value): string
    {
        return preg_replace('/\D+/', '', $value);
    }

    private function documentosObrigatorios(?string $tipoCliente): array
    {
        $normalized = strtoupper(trim((string) $tipoCliente));
        if ($normalized === '') {
            return [];
        }
        if (str_contains($normalized, 'PJ') || str_contains($normalized, 'JURID')) {
            return [
                'RG/CNH',
                'Certidao de estado civil',
                'Comprovante de endereco',
                'IRPF',
                'Contrato Social',
                'Faturamento',
                'Balanco',
                'DRE',
            ];
        }
        if (str_contains($normalized, 'PF') || str_contains($normalized, 'FISIC')) {
            return [
                'RG/CNH',
                'Certidao de estado civil',
                'Comprovante de endereco',
                'IRPF',
                'Extratos',
                'Holerites/INSS',
                'Matricula',
                'IPTU',
            ];
        }
        return [];
    }

    private function validateDocumentosObrigatorios(array $requiredDocs, $documentos): array
    {
        $map = [];
        foreach ($documentos as $documento) {
            $key = $this->normalizeDocKey($documento->tipo ?? '');
            if ($key === '') {
                continue;
            }
            if (!isset($map[$key])) {
                $map[$key] = [];
            }
            $map[$key][] = $documento;
        }

        $missing = [];
        $pending = [];
        foreach ($requiredDocs as $label) {
            $key = $this->normalizeDocKey($label);
            $items = $map[$key] ?? [];
            if (!$items) {
                $missing[] = $label;
                continue;
            }
            $validado = false;
            foreach ($items as $item) {
                if ($this->isDocumentoValidado($item)) {
                    $validado = true;
                    break;
                }
            }
            if (!$validado) {
                $pending[] = $label;
            }
        }

        return [$missing, $pending];
    }

    private function normalizeDocKey(?string $value): string
    {
        $value = strtolower(trim((string) $value));
        return preg_replace('/[^a-z0-9]/', '', $value);
    }

    private function isDocumentoValidado(PropostaDocumento $documento): bool
    {
        $status = strtolower(trim((string) ($documento->status_validacao ?? $documento->status ?? '')));
        if ($status === '') {
            return false;
        }
        return str_contains($status, 'valid') || str_contains($status, 'aprov');
    }

    private function buildParceiroPayload(PropostaCredito $proposta, $documentosRows = null): array
    {
        $cliente = $proposta->cliente;
        $documentosRows = $documentosRows ?: PropostaDocumento::query()
            ->where('proposta_id', $proposta->id)
            ->orderBy('id')
            ->get();
        $documentos = $this->mapDocumentos($documentosRows);

        return [
            'proposta' => [
                'id' => $proposta->id,
                'numero' => $this->formatPropostaNumero($proposta->id),
                'valor_solicitado' => $proposta->valor_solicitado,
                'prazo_meses' => $proposta->prazo_meses,
                'finalidade' => $proposta->finalidade,
                'status' => $proposta->status ?? null,
            ],
            'cliente' => [
                'id' => $cliente?->id,
                'nome' => $cliente?->nome_razao ?? $cliente?->nome,
                'documento' => $cliente?->cpf_cnpj,
                'tipo' => $cliente?->tipo_cliente,
            ],
            'documentos' => $documentos,
        ];
    }

    private function loadEndereco(?int $clienteId): ?array
    {
        if (!$clienteId || !Schema::hasTable('enderecos')) {
            return null;
        }

        $query = DB::table('enderecos')->where('cliente_id', $clienteId);
        if (Schema::hasColumn('enderecos', 'tipo_endereco')) {
            $query->orderByRaw(
                "CASE tipo_endereco
                    WHEN 'PRINCIPAL' THEN 0
                    WHEN 'RESIDENCIAL' THEN 1
                    WHEN 'EMPRESA' THEN 2
                    ELSE 3
                END"
            );
        }
        if (Schema::hasColumn('enderecos', 'id')) {
            $query->orderByDesc('id');
        }

        $row = $query->first();
        return $row ? (array) $row : null;
    }

    private function loadContato(?int $clienteId): ?array
    {
        if (!$clienteId || !Schema::hasTable('contatos')) {
            return null;
        }

        $query = DB::table('contatos')->where('cliente_id', $clienteId);
        if (Schema::hasColumn('contatos', 'id')) {
            $query->orderByDesc('id');
        }

        $row = $query->first();
        return $row ? (array) $row : null;
    }

    private function mapDocumentos($documentos): array
    {
        $items = [];
        foreach ($documentos as $documento) {
            $items[] = [
                'id' => $documento->id,
                'tipo' => $documento->tipo,
                'status' => $documento->status,
                'status_validacao' => $documento->status_validacao,
                'arquivo' => $documento->arquivo,
                'url' => $this->documentUrl($documento->arquivo),
                'criado_em' => $documento->created_at,
            ];
        }
        return $items;
    }

    private function mapSolicitacoes($solicitacoes, string $tipo = 'proposta'): array
    {
        $items = [];
        foreach ($solicitacoes as $solicitacao) {
            $items[] = [
                'id' => $solicitacao->id,
                'token' => $solicitacao->token,
                'canal' => $solicitacao->canal,
                'destinatario' => $solicitacao->destinatario,
                'status' => $solicitacao->status,
                'tipo_cliente' => $solicitacao->tipo_cliente,
                'tipo' => $tipo,
                'created_at' => $solicitacao->created_at,
            ];
        }
        return $items;
    }

    private function documentUrl(?string $path): ?string
    {
        if (!$path) {
            return null;
        }
        $path = ltrim($path, '/');
        $path = preg_replace('#^public/#', '', $path);
        $base = rtrim(config('app.url') ?: request()->getSchemeAndHttpHost(), '/');
        if (str_contains($base, '/adm_capital/api')) {
            $base = str_replace('/adm_capital/api', '/adm_capital', $base);
        } elseif (preg_match('#/api$#', $base)) {
            $base = preg_replace('#/api$#', '', $base);
        }

        return $base.'/public/'.$path;
    }

    private function formatPropostaNumero(?int $id): string
    {
        if (!$id) {
            return 'PR-0000';
        }
        return 'PR-'.str_pad((string) $id, 4, '0', STR_PAD_LEFT);
    }

    private function parseApiHeaders(?string $headers): array
    {
        if (!$headers) {
            return [];
        }
        $decoded = json_decode($headers, true);
        if (!is_array($decoded)) {
            return [];
        }
        return $decoded;
    }

    private function sendParceiroApi(Parceiro $parceiro, array $data): array
    {
        if ($parceiro->api_enabled === false) {
            return [
                'ok' => false,
                'status' => 422,
                'body' => 'API do parceiro desativada.',
            ];
        }

        if ($parceiro->tipo_integracao === 'c6_home_equity') {
            return $this->sendC6Api($parceiro, $data);
        }

        $headers = $this->parseApiHeaders($parceiro->api_headers);
        $apiKey = $this->decryptIfNeeded($parceiro->api_key);
        if ($apiKey) {
            $headers['X-API-KEY'] = $apiKey;
        }

        $request = Http::timeout(20)->withHeaders($headers);
        $apiToken = $this->decryptIfNeeded($parceiro->api_token);
        if ($apiToken) {
            $request = $request->withToken($apiToken);
        }

        $response = $request->post($parceiro->api_url, $data);
        if (!$response->successful()) {
            return [
                'ok' => false,
                'status' => $response->status(),
                'body' => $response->body(),
            ];
        }

        return [
            'ok' => true,
            'status' => $response->status(),
            'body' => $response->json() ?? $response->body(),
        ];
    }

    private function sendC6Api(Parceiro $parceiro, array $data): array
    {
        $payload = $this->buildC6Payload($parceiro, $data);
        if ($payload['error']) {
            return [
                'ok' => false,
                'status' => 422,
                'body' => $payload['error'],
            ];
        }

        $token = $this->getC6Token($parceiro);
        if (!$token) {
            return [
                'ok' => false,
                'status' => 502,
                'body' => 'Falha ao obter token C6.',
            ];
        }

        $url = $parceiro->api_url ?: ($parceiro->api_base_url ? rtrim($parceiro->api_base_url, '/').'/home-equity/simulations/partner' : null);
        if (!$url) {
            return [
                'ok' => false,
                'status' => 422,
                'body' => 'API C6 nao configurada.',
            ];
        }

        $headers = $this->parseApiHeaders($parceiro->api_headers);
        $headers['Content-Type'] = $headers['Content-Type'] ?? 'text/plain';

        $response = Http::timeout(25)
            ->withHeaders($headers)
            ->withToken($token)
            ->post($url, $payload['data']);

        if (!$response->successful()) {
            return [
                'ok' => false,
                'status' => $response->status(),
                'body' => $response->body(),
            ];
        }

        return [
            'ok' => true,
            'status' => $response->status(),
            'body' => $response->json() ?? $response->body(),
        ];
    }

    private function buildC6Payload(Parceiro $parceiro, array $data): array
    {
        $proposta = $data['proposta'] ?? [];
        $cliente = $data['cliente'] ?? [];
        $endereco = $data['endereco'] ?? $this->loadEndereco($cliente['id'] ?? null);
        $contato = $data['contato'] ?? $this->loadContato($cliente['id'] ?? null);

        $external = $this->loadExternalImobiliarioByCliente($cliente['documento'] ?? null);

        $gender = $this->mapGender($external->genero ?? $external->sexo ?? null);
        $phone = $this->parsePhone($contato['telefone'] ?? $contato['celular'] ?? $external->telefone ?? null);

        $missing = [];
        if (!$cliente['documento']) {
            $missing[] = 'CPF';
        }
        if (!$cliente['nome']) {
            $missing[] = 'Nome';
        }
        if (!$gender) {
            $missing[] = 'Genero';
        }
        if (!$cliente['email'] && empty($external->email)) {
            $missing[] = 'Email';
        }
        if (!$phone) {
            $missing[] = 'Telefone';
        }
        if (!$endereco) {
            $missing[] = 'Endereco';
        }
        if (!$external) {
            $missing[] = 'Dados do formulario (imovel)';
        }
        if ($missing) {
            return ['data' => null, 'error' => 'Dados obrigatorios ausentes: '.implode(', ', $missing).'.'];
        }

        $payload = [
            'person' => [
                'cpf_number' => $this->normalizeDocument((string) $cliente['documento']),
                'name' => $cliente['nome'],
                'gender' => $gender,
                'email' => $cliente['email'] ?? $external->email ?? null,
                'phone' => $phone,
            ],
            'property' => [
                'type' => $this->mapPropertyType($external->tipo_imovel ?? null),
                'owner' => $this->mapOwner($external->propriedade ?? null),
                'estimated_value' => $external->valor_imovel ?? null,
                'address' => [
                    'street' => $endereco['logradouro'] ?? $external->endereco ?? null,
                    'number' => $endereco['numero'] ?? $external->numero ?? null,
                    'neighborhood' => $endereco['bairro'] ?? $external->bairro ?? null,
                    'city' => $endereco['cidade'] ?? $external->cidade ?? null,
                    'state' => $endereco['uf'] ?? $external->estado ?? null,
                    'zip_code' => $this->normalizeDocument((string) ($endereco['cep'] ?? $external->cep ?? '')),
                ],
                'payoff_amount' => $external->saldo_financiado ?? null,
                'footage' => $external->metragem ?? null,
                'in_condominium' => $this->normalizeBool($external->condominio ?? null),
            ],
            'payment' => [
                'amount_value' => $proposta['valor_solicitado'] ?? $external->valor_emprestimo ?? null,
                'amortization_period' => $proposta['prazo_meses'] ?? $external->prazo ?? null,
                'amortization_type' => 'PRICE',
                'interest_installment' => 'FIXED',
                'dilute_registry_costs' => true,
                'dilute_iof' => true,
                'dilute_property_valuation' => true,
            ],
            'partner_cnpj' => $parceiro->cnpj,
        ];

        $payload = $this->applyPayloadMapOverrides($payload, $parceiro->api_payload_map);

        return ['data' => $payload, 'error' => null];
    }

    private function getC6Token(Parceiro $parceiro): ?string
    {
        $expiresAt = $parceiro->api_token_expires_at ? Carbon::parse($parceiro->api_token_expires_at) : null;
        $currentToken = $this->decryptIfNeeded($parceiro->api_token);
        if ($currentToken && $expiresAt && $expiresAt->isFuture()) {
            return $currentToken;
        }

        $clientId = $this->decryptIfNeeded($parceiro->api_client_id);
        $clientSecret = $this->decryptIfNeeded($parceiro->api_client_secret);
        if (!$clientId || !$clientSecret) {
            return null;
        }

        $authUrl = $parceiro->api_auth_url ?: ($parceiro->api_base_url ? rtrim($parceiro->api_base_url, '/').'/home-equity/token/partner' : null);
        if (!$authUrl) {
            return null;
        }

        $response = Http::asForm()
            ->timeout(20)
            ->post($authUrl, [
                'grant_type' => 'client_credentials',
                'client_id' => $clientId,
                'client_secret' => $clientSecret,
            ]);

        if (!$response->successful()) {
            return null;
        }

        $json = $response->json();
        $token = $json['access_token'] ?? null;
        $expiresIn = (int) ($json['expires_in'] ?? 0);
        if (!$token) {
            return null;
        }

        $parceiro->api_token = Crypt::encryptString($token);
        if ($expiresIn > 0) {
            $parceiro->api_token_expires_at = Carbon::now()->addSeconds($expiresIn - 60);
        }
        $parceiro->save();

        return $token;
    }

    private function loadExternalImobiliarioByCliente(?string $document): ?object
    {
        $document = $this->normalizeDocument((string) $document);
        if ($document === '') {
            return null;
        }
        try {
            return DB::connection(self::EXTERNAL_CONNECTION)
                ->table('simulacoes_imobiliarias as s')
                ->select([
                    's.id',
                    's.nome',
                    's.email',
                    's.telefone',
                    's.cpf',
                    's.cep',
                    's.endereco',
                    's.numero',
                    's.bairro',
                    's.cidade',
                    's.estado',
                    's.valor_imovel',
                    's.tipo_imovel',
                    's.propriedade',
                    's.metragem',
                    's.condominio',
                    's.saldo_financiado',
                    's.valor_emprestimo',
                    's.prazo',
                    's.genero',
                    's.sexo',
                ])
                ->whereRaw(
                    "REPLACE(REPLACE(REPLACE(REPLACE(s.cpf, '.', ''), '-', ''), '/', ''), ' ', '') = ?",
                    [$document]
                )
                ->orderByDesc('s.id')
                ->first();
        } catch (\Throwable $exception) {
            return null;
        }
    }

    private function mapGender(?string $value): ?string
    {
        $text = strtolower(trim((string) $value));
        if ($text === '') {
            return null;
        }
        if (str_contains($text, 'fem')) {
            return 'FEMALE';
        }
        if (str_contains($text, 'masc')) {
            return 'MALE';
        }
        return null;
    }

    private function mapPropertyType(?string $value): string
    {
        $text = strtoupper(trim((string) $value));
        if ($text === '') {
            return 'RESIDENTIAL';
        }
        if (str_contains($text, 'APART')) {
            return 'APARTMENT';
        }
        if (str_contains($text, 'CASA')) {
            return 'HOUSE';
        }
        if (str_contains($text, 'TERRE')) {
            return 'LAND';
        }
        if (str_contains($text, 'SALA')) {
            return 'COMMERCIAL_ROOM';
        }
        if (str_contains($text, 'COMERC')) {
            return 'BUSINESS';
        }
        return 'RESIDENTIAL';
    }

    private function mapOwner(?string $value): string
    {
        $text = strtoupper(trim((string) $value));
        if ($text === '') {
            return 'SELF';
        }
        if (str_contains($text, 'FILH')) {
            return 'SONS';
        }
        if (str_contains($text, 'PAI') || str_contains($text, 'MAE')) {
            return 'PARENTS';
        }
        if (str_contains($text, 'PARCEIRO')) {
            return 'PARTNER';
        }
        if (str_contains($text, 'PROPRIO') || str_contains($text, 'SELF')) {
            return 'SELF';
        }
        return 'SELF';
    }

    private function parsePhone(?string $value): ?array
    {
        if (!$value) {
            return null;
        }
        $digits = preg_replace('/\D+/', '', $value);
        if (!$digits) {
            return null;
        }
        if (strlen($digits) >= 10) {
            $ddd = substr($digits, 0, 2);
            $number = substr($digits, 2);
            return [
                'area_code' => (int) $ddd,
                'number' => (int) $number,
            ];
        }
        return null;
    }

    private function normalizeBool($value): bool
    {
        if (is_bool($value)) {
            return $value;
        }
        $text = strtolower(trim((string) $value));
        return in_array($text, ['1', 'true', 'sim', 'yes'], true);
    }

    private function applyPayloadMapOverrides(array $payload, ?string $mapJson): array
    {
        if (!$mapJson) {
            return $payload;
        }
        $map = json_decode($mapJson, true);
        if (!is_array($map)) {
            return $payload;
        }
        foreach ($map as $path => $value) {
            $payload = $this->setArrayValue($payload, $path, $value);
        }
        return $payload;
    }

    private function setArrayValue(array $data, string $path, $value): array
    {
        $parts = array_filter(explode('.', $path), fn ($part) => $part !== '');
        if (!$parts) {
            return $data;
        }
        $ref =& $data;
        foreach ($parts as $index => $part) {
            if ($index === count($parts) - 1) {
                $ref[$part] = $value;
            } else {
                if (!isset($ref[$part]) || !is_array($ref[$part])) {
                    $ref[$part] = [];
                }
                $ref =& $ref[$part];
            }
        }
        return $data;
    }

    private function decryptIfNeeded(?string $value): ?string
    {
        if (!$value) {
            return null;
        }
        try {
            return Crypt::decryptString($value);
        } catch (\Throwable $exception) {
            return $value;
        }
    }

    private function sendParceiroEmail(Parceiro $parceiro, array $data): void
    {
        $propostaNumero = $data['proposta']['numero'] ?? $this->formatPropostaNumero($data['proposta']['id'] ?? null);
        $clienteNome = $data['cliente']['nome'] ?? 'Cliente';
        $subject = 'LPC Capital - Proposta '.$propostaNumero.' - '.$clienteNome;

        $html = $this->buildParceiroEmailHtml($parceiro, $data);
        $text = $this->buildParceiroEmailText($parceiro, $data);

        $fromAddress = config('mail.from.address') ?: env('MAIL_FROM_ADDRESS');
        $fromName = config('mail.from.name') ?: env('MAIL_FROM_NAME');

        Mail::send([], [], function ($message) use ($parceiro, $subject, $fromAddress, $fromName, $html, $text) {
            if ($fromAddress) {
                $message->from($fromAddress, $fromName ?: null);
            }
            $message->to($parceiro->contato_email)
                ->replyTo('contato@marianalpccapital.com.br', 'LPC Capital')
                ->subject($subject)
                ->html($html)
                ->text($text);
        });
    }

    private function buildParceiroEmailHtml(Parceiro $parceiro, array $data): string
    {
        $cliente = $data['cliente'] ?? [];
        $proposta = $data['proposta'] ?? [];
        $documentos = $data['documentos'] ?? [];
        $lista = '';
        foreach ($documentos as $doc) {
            $url = $doc['url'] ?? '';
            $label = $doc['tipo'] ?? '-';
            $linha = $url ? '<a href="'.e($url).'">'.e($label).'</a>' : e($label);
            $lista .= '<li>'.$linha.'</li>';
        }

        $html = '<!doctype html><html><head><meta charset="utf-8">';
        $html .= '<style>body{font-family:Arial,sans-serif;color:#0f172a;margin:24px}';
        $html .= 'h1{font-size:22px;margin:0}h2{font-size:13px;margin:18px 0 8px;text-transform:uppercase;letter-spacing:.12em;color:#475569}';
        $html .= 'ul{padding-left:18px}li{margin-bottom:6px;font-size:13px}.card{border:1px solid #e2e8f0;border-radius:12px;padding:16px;margin-top:16px}';
        $html .= 'a{color:#0ea5e9;text-decoration:none}</style></head><body>';
        $html .= '<h1>Proposta '.$this->formatEmailValue($proposta['numero'] ?? '').'</h1>';
        $html .= '<p>Parceiro: '.$this->formatEmailValue($parceiro->nome).'</p>';
        $html .= '<div class="card"><h2>Dados do cliente</h2>';
        $html .= '<p><strong>Nome:</strong> '.$this->formatEmailValue($cliente['nome'] ?? '').'</p>';
        $html .= '<p><strong>Documento:</strong> '.$this->formatEmailValue($cliente['documento'] ?? '').'</p>';
        $html .= '<p><strong>Tipo:</strong> '.$this->formatEmailValue($cliente['tipo'] ?? '').'</p></div>';
        $html .= '<div class="card"><h2>Resumo da proposta</h2>';
        $html .= '<p><strong>Valor solicitado:</strong> '.$this->formatEmailValue($this->formatCurrency($proposta['valor_solicitado'] ?? null)).'</p>';
        $html .= '<p><strong>Prazo:</strong> '.$this->formatEmailValue($proposta['prazo_meses'] ?? null).'</p>';
        $html .= '<p><strong>Finalidade:</strong> '.$this->formatEmailValue($proposta['finalidade'] ?? '').'</p></div>';
        $html .= '<div class="card"><h2>Documentos enviados</h2><ul>'.$lista.'</ul></div>';
        $html .= '<p>Equipe LPC Capital</p>';
        $html .= '</body></html>';

        return $html;
    }

    private function buildParceiroEmailText(Parceiro $parceiro, array $data): string
    {
        $cliente = $data['cliente'] ?? [];
        $proposta = $data['proposta'] ?? [];
        $documentos = $data['documentos'] ?? [];

        $lines = [
            'Proposta '.$this->formatEmailValue($proposta['numero'] ?? ''),
            'Parceiro: '.$this->formatEmailValue($parceiro->nome),
            '',
            'Dados do cliente:',
            'Nome: '.$this->formatEmailValue($cliente['nome'] ?? ''),
            'Documento: '.$this->formatEmailValue($cliente['documento'] ?? ''),
            'Tipo: '.$this->formatEmailValue($cliente['tipo'] ?? ''),
            '',
            'Resumo da proposta:',
            'Valor solicitado: '.$this->formatEmailValue($this->formatCurrency($proposta['valor_solicitado'] ?? null)),
            'Prazo (meses): '.$this->formatEmailValue($proposta['prazo_meses'] ?? null),
            'Finalidade: '.$this->formatEmailValue($proposta['finalidade'] ?? ''),
            '',
            'Documentos enviados:',
        ];

        foreach ($documentos as $doc) {
            $label = $doc['tipo'] ?? '-';
            $url = $doc['url'] ?? '';
            $lines[] = $url ? '- '.$label.' ('.$url.')' : '- '.$label;
        }

        $lines[] = '';
        $lines[] = 'Equipe LPC Capital';

        return implode("\n", $lines);
    }

    private function formatEmailValue($value): string
    {
        if ($value === null) {
            return '-';
        }
        $text = trim((string) $value);
        return $text === '' ? '-' : $text;
    }

    private function formatCurrency($value): string
    {
        if ($value === null || $value === '') {
            return '-';
        }
        $number = is_numeric($value) ? (float) $value : null;
        if ($number === null) {
            return (string) $value;
        }
        return 'R$ '.number_format($number, 2, ',', '.');
    }

    private function resolveDefaultStatusId(): ?int
    {
        try {
            if (!Schema::hasTable('status_proposta')) {
                return null;
            }
        } catch (\Throwable $exception) {
            return null;
        }

        $preferred = DB::table('status_proposta')
            ->select(['id', 'nome'])
            ->orderBy('id')
            ->get();

        if ($preferred->isEmpty()) {
            return null;
        }

        foreach ($preferred as $row) {
            $name = strtolower(trim($row->nome ?? ''));
            if (in_array($name, ['rascunho', 'novo', 'nova'], true)) {
                return (int) $row->id;
            }
        }

        return isset($preferred[0]) ? (int) $preferred[0]->id : null;
    }
}
