<?php

namespace App\Http\Controllers;

use App\Models\PropostaCredito;
use App\Models\PropostaDocumento;
use App\Models\PropostaDocumentoSolicitacao;
use App\Models\Configuracao;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class PropostaDocumentosController extends Controller
{
    private const DEFAULT_SMS_TEXTO_PROPOSTA = "LPC Capital - documentos da proposta\nOla {nome}\nObjetivo: {objetivo}\nDocumentos: {documentos}\nEnvio: {link}";

    private const DOCUMENTOS = [
        'PF' => [
            ['key' => 'rg_cnh', 'label' => 'RG/CNH'],
            ['key' => 'certidao_estado_civil', 'label' => 'Certidao de estado civil'],
            ['key' => 'comprovante_endereco', 'label' => 'Comprovante de endereco'],
            ['key' => 'irpf', 'label' => 'IRPF'],
            ['key' => 'extratos', 'label' => 'Extratos'],
            ['key' => 'holerites_inss', 'label' => 'Holerites/INSS'],
            ['key' => 'matricula', 'label' => 'Matricula'],
            ['key' => 'iptu', 'label' => 'IPTU'],
        ],
        'PJ' => [
            ['key' => 'rg_cnh', 'label' => 'RG/CNH'],
            ['key' => 'certidao_estado_civil', 'label' => 'Certidao de estado civil'],
            ['key' => 'comprovante_endereco', 'label' => 'Comprovante de endereco'],
            ['key' => 'irpf', 'label' => 'IRPF'],
            ['key' => 'contrato_social', 'label' => 'Contrato Social'],
            ['key' => 'faturamento', 'label' => 'Faturamento'],
            ['key' => 'balanco', 'label' => 'Balanco'],
            ['key' => 'dre', 'label' => 'DRE'],
        ],
    ];

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->requireRole(['admin', 'gestor']);
            return $next($request);
        })->except(['show', 'upload']);
    }

    public function store(Request $request, PropostaCredito $proposta)
    {
        $payload = $request->validate([
            'canal' => ['required', 'string', 'in:email,whatsapp,sms'],
            'destinatario' => ['required', 'string', 'max:255'],
            'tipo_cliente' => ['nullable', 'string', 'in:PF,PJ'],
        ]);

        $cliente = $proposta->cliente;
        $tipoCliente = $payload['tipo_cliente'] ?: ($cliente->tipo_cliente ?? null);
        if (!$tipoCliente || !isset(self::DOCUMENTOS[$tipoCliente])) {
            return response()->json(['message' => 'Tipo de cliente invalido.'], 422);
        }

        $destinatario = trim($payload['destinatario']);
        $invalidEmails = [];
        $validEmails = [];
        if ($payload['canal'] === 'email') {
            $emails = $this->parseEmailList($destinatario);
            if (!$emails) {
                return response()->json(['message' => 'Informe ao menos um email valido.'], 422);
            }
            $invalidEmails = $this->invalidEmails($emails);
            $validEmails = array_values(array_diff($emails, $invalidEmails));
            if (!$validEmails) {
                $list = $invalidEmails ? implode(', ', $invalidEmails) : $destinatario;
                return response()->json(['message' => 'Nenhum email valido informado. Invalidos: '.$list], 422);
            }
            $destinatario = implode(', ', $validEmails);
        } else {
            $phone = $this->normalizePhone($destinatario);
            if (!$phone) {
                $label = $payload['canal'] === 'whatsapp' ? 'WhatsApp' : 'telefone';
                return response()->json(['message' => 'Informe um '.$label.' valido.'], 422);
            }
            $destinatario = $phone;
        }

        $token = Str::random(48);
        $nomeCliente = $cliente->nome_razao ?? $cliente->nome ?? null;
        $objetivo = $proposta->finalidade ?? null;

        $solicitacao = PropostaDocumentoSolicitacao::query()->create([
            'proposta_id' => $proposta->id,
            'token' => $token,
            'canal' => $payload['canal'],
            'destinatario' => $destinatario,
            'tipo_cliente' => $tipoCliente,
            'nome_cliente' => $nomeCliente,
            'objetivo' => $objetivo,
            'status' => 'enviada',
        ]);

        $link = $this->buildSolicitacaoLink($token);
        $documentos = self::DOCUMENTOS[$tipoCliente];
        $whatsapp = null;
        $sms = null;

        if ($payload['canal'] === 'email') {
            $emailError = null;
            $sentEmails = [];
            $failedEmails = [];
            try {
                $result = $this->sendEmailSolicitacao($validEmails, $proposta, $nomeCliente, $objetivo, $documentos, $link);
                $sentEmails = $result['sent'] ?? [];
                $failedEmails = $result['failed'] ?? [];
            } catch (\Throwable $exception) {
                $emailError = $exception->getMessage();
            }

            $emailSent = count($sentEmails) > 0;
            if (!$emailSent) {
                $solicitacao->status = 'erro_envio';
                $solicitacao->save();
            }

            $message = $emailSent
                ? 'Solicitacao enviada com sucesso.'
                : 'Solicitacao criada, mas nenhum email foi enviado. Use o link gerado.';

            if ($emailSent && ($invalidEmails || $failedEmails)) {
                $message = 'Solicitacao enviada para '.count($sentEmails).' email(s).';
            }
            if ($invalidEmails) {
                $message .= ' Invalidos: '.implode(', ', $invalidEmails).'.';
            }
            if ($failedEmails) {
                $message .= ' Falha no envio: '.implode(', ', $failedEmails).'.';
            }

            if ($emailSent && !$invalidEmails && !$failedEmails) {
                Log::info('Solicitacao de documentos (proposta) enviada por email.', [
                    'proposta_id' => $proposta->id,
                    'destinatario' => $destinatario,
                    'token' => $solicitacao->token,
                ]);
            } else {
                Log::warning('Falha ao enviar email da solicitacao de proposta.', [
                    'proposta_id' => $proposta->id,
                    'destinatario' => $destinatario,
                    'token' => $solicitacao->token,
                    'invalidos' => $invalidEmails,
                    'falhas' => $failedEmails,
                    'error' => $emailError,
                ]);
            }
            return response()->json([
                'message' => $message,
                'link' => $link,
                'email_sent' => $emailSent,
                'invalid_emails' => $invalidEmails,
                'failed_emails' => $failedEmails,
                'sent_emails' => $sentEmails,
                'token' => $solicitacao->token,
            ]);
        } elseif ($payload['canal'] === 'whatsapp') {
            $message = $this->buildWhatsappMessage($nomeCliente, $objetivo, $documentos, $link);
            $whatsapp = $this->sendWhatsappMessage($destinatario, $message);
        } else {
            $message = $this->buildSmsMessage($nomeCliente, $objetivo, $documentos, $link);
            $sms = $this->sendSmsMessage($destinatario, $message);
        }

        return response()->json([
            'message' => 'Solicitacao enviada com sucesso.',
            'link' => $link,
            'whatsapp_sent' => $whatsapp['sent'] ?? null,
            'whatsapp_link' => $whatsapp['link'] ?? null,
            'sms_sent' => $sms['sent'] ?? null,
            'token' => $solicitacao->token,
        ]);
    }
    public function show(string $token)
    {
        $solicitacao = PropostaDocumentoSolicitacao::query()
            ->with('propostaCredito.cliente')
            ->where('token', $token)
            ->first();

        if (!$solicitacao) {
            return response()->json(['message' => 'Solicitacao nao encontrada.'], 404);
        }

        $proposta = $solicitacao->propostaCredito;
        $cliente = $proposta?->cliente;
        $tipoCliente = $solicitacao->tipo_cliente ?: ($cliente->tipo_cliente ?? null);
        $documentos = $tipoCliente && isset(self::DOCUMENTOS[$tipoCliente]) ? self::DOCUMENTOS[$tipoCliente] : [];

        return response()->json([
            'token' => $solicitacao->token,
            'status' => $solicitacao->status,
            'proposta_id' => $solicitacao->proposta_id,
            'cliente' => [
                'nome' => $solicitacao->nome_cliente ?? ($cliente->nome_razao ?? $cliente->nome ?? '-'),
                'tipo' => $tipoCliente,
                'documento' => $cliente->cpf_cnpj ?? null,
            ],
            'objetivo' => $solicitacao->objetivo ?? ($proposta?->finalidade ?? null),
            'documentos' => $documentos,
        ]);
    }

    public function upload(Request $request, string $token)
    {
        $solicitacao = PropostaDocumentoSolicitacao::query()
            ->where('token', $token)
            ->first();

        if (!$solicitacao) {
            return response()->json(['message' => 'Solicitacao nao encontrada.'], 404);
        }

        $files = $request->file('documentos', []);
        if (!$files) {
            return response()->json(['message' => 'Envie ao menos um documento.'], 422);
        }

        $tipoCliente = $solicitacao->tipo_cliente;
        $docMap = $this->documentosMap($tipoCliente);
        $saved = [];

        foreach ($files as $docKey => $docFiles) {
            if (!$docFiles) {
                continue;
            }
            $docFiles = is_array($docFiles) ? $docFiles : [$docFiles];
            foreach ($docFiles as $file) {
                if (!$file || !$file->isValid()) {
                    continue;
                }
                $path = $this->storeFile($file, $solicitacao->proposta_id);
                $label = $docMap[$docKey] ?? $docKey;
                PropostaDocumento::query()->create([
                    'proposta_id' => $solicitacao->proposta_id,
                    'tipo' => $label,
                    'arquivo' => $path,
                    'status' => 'recebido',
                    'arquivo_path' => $path,
                    'status_validacao' => 'recebido',
                ]);
                $saved[] = [
                    'tipo' => $label,
                    'arquivo' => $path,
                ];
            }
        }

        if (!$saved) {
            return response()->json(['message' => 'Nenhum documento valido encontrado.'], 422);
        }

        $solicitacao->status = 'recebido';
        $solicitacao->save();

        return response()->json([
            'message' => 'Documentos enviados com sucesso.',
            'files' => $saved,
        ]);
    }

    private function documentosMap(?string $tipoCliente): array
    {
        if (!$tipoCliente || !isset(self::DOCUMENTOS[$tipoCliente])) {
            return [];
        }
        $map = [];
        foreach (self::DOCUMENTOS[$tipoCliente] as $doc) {
            $map[$doc['key']] = $doc['label'];
        }
        return $map;
    }

    private function buildSolicitacaoLink(string $token): string
    {
        $host = request()->getSchemeAndHttpHost();
        return $host.'/formulario/documentos/?token='.$token;
    }

    private function sendEmailSolicitacao(
        array $emails,
        PropostaCredito $proposta,
        ?string $nomeCliente,
        ?string $objetivo,
        array $documentos,
        string $link
    ): array {
        $propostaNumero = $this->formatPropostaNumero($proposta->id);
        $subject = 'LPC Capital - Documentos da proposta '.$propostaNumero;
        if ($nomeCliente) {
            $subject .= ' - '.$nomeCliente;
        }

        $html = $this->buildSolicitacaoEmailHtml($propostaNumero, $nomeCliente, $objetivo, $documentos, $link);
        $text = $this->buildSolicitacaoEmailText($propostaNumero, $nomeCliente, $objetivo, $documentos, $link);

        $fromAddress = config('mail.from.address') ?: env('MAIL_FROM_ADDRESS');
        $fromName = config('mail.from.name') ?: env('MAIL_FROM_NAME');
        if ($fromAddress && !filter_var($fromAddress, FILTER_VALIDATE_EMAIL)) {
            $fromAddress = null;
        }
        if (!$fromAddress) {
            $fromAddress = 'contato@marianalpccapital.com.br';
        }

        $sent = [];
        $failed = [];
        $errors = [];

        foreach ($emails as $email) {
            try {
                Mail::send([], [], function ($message) use ($email, $subject, $fromAddress, $fromName, $html, $text) {
                    if ($fromAddress) {
                        $message->from($fromAddress, $fromName ?: null);
                    }
                    $message->to($email)
                        ->replyTo('contato@marianalpccapital.com.br', 'LPC Capital')
                        ->subject($subject)
                        ->html($html)
                        ->text($text);
                });
                $sent[] = $email;
            } catch (\Throwable $exception) {
                $failed[] = $email;
                $errors[$email] = $exception->getMessage();
            }
        }

        return [
            'sent' => $sent,
            'failed' => $failed,
            'errors' => $errors,
        ];
    }
    private function buildSolicitacaoEmailHtml(
        string $propostaNumero,
        ?string $nomeCliente,
        ?string $objetivo,
        array $documentos,
        string $link
    ): string {
        $title = $nomeCliente ?: 'Cliente';
        $lista = '';
        foreach ($documentos as $doc) {
            $lista .= '<li>'.e($doc['label']).'</li>';
        }

        $html = '<!doctype html><html><head><meta charset="utf-8">';
        $html .= '<style>body{font-family:Arial,sans-serif;color:#0f172a;margin:24px}';
        $html .= 'h1{font-size:22px;margin:0}h2{font-size:13px;margin:18px 0 8px;text-transform:uppercase;letter-spacing:.12em;color:#475569}';
        $html .= 'ul{padding-left:18px}li{margin-bottom:6px;font-size:13px}.card{border:1px solid #e2e8f0;border-radius:12px;padding:16px;margin-top:16px}';
        $html .= 'a{color:#0ea5e9;text-decoration:none}</style></head><body>';
        $html .= '<h1>Documentos pendentes - Proposta '.$propostaNumero.'</h1>';
        $html .= '<p>Ola '.$this->formatEmailValue($title).',</p>';
        $html .= '<p>Para dar sequencia a sua proposta, precisamos dos documentos abaixo.</p>';
        if ($objetivo) {
            $html .= '<p><strong>Objetivo:</strong> '.e($objetivo).'</p>';
        }
        $html .= '<div class="card"><h2>Documentos solicitados</h2><ul>'.$lista.'</ul></div>';
        $html .= '<p>Envie tudo pelo link seguro: <a href="'.e($link).'">Enviar documentos</a></p>';
        $html .= '<p>Se tiver duvidas, responda este e-mail ou fale com seu consultor.</p>';
        $html .= '<p>Equipe LPC Capital</p>';
        $html .= '</body></html>';

        return $html;
    }

    private function buildSolicitacaoEmailText(
        string $propostaNumero,
        ?string $nomeCliente,
        ?string $objetivo,
        array $documentos,
        string $link
    ): string {
        $lines = [
            'Documentos pendentes - Proposta '.$propostaNumero,
            'Cliente: '.$this->formatEmailValue($nomeCliente),
        ];
        if ($objetivo) {
            $lines[] = 'Objetivo: '.$objetivo;
        }
        $lines[] = '';
        $lines[] = 'Documentos solicitados:';
        foreach ($documentos as $doc) {
            $lines[] = '- '.$doc['label'];
        }
        $lines[] = '';
        $lines[] = 'Envie os documentos pelo link: '.$link;
        $lines[] = 'Se tiver duvidas, responda este e-mail ou fale com seu consultor.';
        $lines[] = '';
        $lines[] = 'Equipe LPC Capital';

        return implode("\n", $lines);
    }

    private function buildWhatsappMessage(
        ?string $nomeCliente,
        ?string $objetivo,
        array $documentos,
        string $link
    ): string {
        $lines = [
            'LPC Capital - documentos da proposta',
        ];
        if ($nomeCliente) {
            $lines[] = 'Ola '.$nomeCliente.',';
        }
        $lines[] = 'Para avancar com sua proposta, precisamos dos documentos abaixo:';
        if ($objetivo) {
            $lines[] = 'Objetivo: '.$objetivo;
        }
        $lines[] = 'Documentos solicitados:';
        foreach ($documentos as $doc) {
            $lines[] = '- '.$doc['label'];
        }
        $lines[] = 'Envio: '.$link;
        $lines[] = 'Se precisar, fale com a equipe LPC Capital.';

        return implode("\n", $lines);
    }

    private function buildSmsMessage(
        ?string $nomeCliente,
        ?string $objetivo,
        array $documentos,
        string $link
    ): string {
        $labels = [];
        foreach ($documentos as $doc) {
            $labels[] = $doc['label'];
        }
        $config = $this->smsConfig();
        $template = $config['texto_proposta'] ?: self::DEFAULT_SMS_TEXTO_PROPOSTA;

        return $this->renderSmsTemplate($template, [
            'nome' => $nomeCliente ?? '',
            'objetivo' => $objetivo ?? '',
            'documentos' => $labels ? implode(', ', $labels) : '',
            'link' => $link,
        ]);
    }

    private function renderSmsTemplate(string $template, array $data): string
    {
        $replacements = [
            '{nome}' => $data['nome'] ?? '',
            '{objetivo}' => $data['objetivo'] ?? '',
            '{documentos}' => $data['documentos'] ?? '',
            '{link}' => $data['link'] ?? '',
        ];

        $message = strtr($template, $replacements);
        $lines = preg_split('/\r?\n/', $message) ?: [];
        $filtered = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '') {
                continue;
            }
            if (str_ends_with($line, ':')) {
                continue;
            }
            $filtered[] = $line;
        }

        return implode("\n", $filtered);
    }

    private function sendWhatsappMessage(string $phone, string $message): array
    {
        $config = $this->whatsappConfig();
        $apiUrl = $config['api_url'];
        if (!$apiUrl) {
            return [
                'sent' => false,
                'link' => $this->buildWhatsappLink($phone, $message),
            ];
        }

        $payload = [
            'phone' => $phone,
            'message' => $message,
        ];
        $apiKey = $config['api_key'];
        if ($apiKey) {
            $payload['apikey'] = $apiKey;
        }

        $request = Http::timeout(12);
        $token = $config['api_token'];
        if ($token) {
            $request = $request->withToken($token);
        }

        try {
            $response = $request->post($apiUrl, $payload);
        } catch (\Throwable $exception) {
            return [
                'sent' => false,
                'link' => $this->buildWhatsappLink($phone, $message),
            ];
        }

        if (!$response->successful()) {
            return [
                'sent' => false,
                'link' => $this->buildWhatsappLink($phone, $message),
            ];
        }

        return [
            'sent' => true,
            'link' => $this->buildWhatsappLink($phone, $message),
        ];
    }

    private function sendSmsMessage(string $phone, string $message): array
    {
        $config = $this->smsConfig();
        $apiUrl = $config['api_url'];
        $authKey = $config['auth_key'];
        if (!$apiUrl || !$authKey) {
            return [
                'sent' => false,
            ];
        }

        $payload = [
            'Receivers' => $phone,
            'Content' => $message,
        ];
        if ($config['sender']) {
            $payload['Sender'] = $config['sender'];
        }

        try {
            $response = Http::timeout(12)
                ->withHeaders([
                    'auth-key' => $authKey,
                    'Content-Type' => 'application/json',
                ])
                ->post($apiUrl, $payload);
        } catch (\Throwable $exception) {
            return [
                'sent' => false,
            ];
        }

        if (!$response->successful()) {
            return [
                'sent' => false,
            ];
        }

        $body = $response->json();
        if (is_array($body) && array_key_exists('Success', $body) && !$body['Success']) {
            return [
                'sent' => false,
            ];
        }

        return [
            'sent' => true,
        ];
    }

    private function buildWhatsappLink(string $phone, string $message): string
    {
        $encoded = urlencode($message);
        return 'https://wa.me/'.$phone.'?text='.$encoded;
    }

    private function whatsappConfig(): array
    {
        $config = Configuracao::query()->first();

        return [
            'api_url' => $config?->whatsapp_api_url ?: env('WHATSAPP_API_URL'),
            'api_token' => $config?->whatsapp_api_token ?: env('WHATSAPP_API_TOKEN'),
            'api_key' => $config?->whatsapp_api_key ?: env('WHATSAPP_API_KEY'),
        ];
    }

    private function smsConfig(): array
    {
        $config = Configuracao::query()->first();
        $defaultUrl = 'https://sms.comtele.com.br/api/v2/send';

        return [
            'api_url' => $config?->sms_api_url ?: env('COMTELE_SMS_API_URL', $defaultUrl),
            'auth_key' => $config?->sms_auth_key ?: env('COMTELE_SMS_AUTH_KEY'),
            'sender' => $config?->sms_sender ?: env('COMTELE_SMS_SENDER'),
            'texto_proposta' => $config?->sms_texto_proposta ?: env('COMTELE_SMS_TEXTO_PROPOSTA'),
        ];
    }

    private function formatPropostaNumero(?int $id): string
    {
        if (!$id) {
            return 'PR-0000';
        }
        return 'PR-'.str_pad((string) $id, 4, '0', STR_PAD_LEFT);
    }

    private function normalizePhone(string $value): ?string
    {
        $digits = preg_replace('/\D+/', '', $value);
        if (!$digits) {
            return null;
        }
        if (strlen($digits) < 10) {
            return null;
        }
        return $digits;
    }

    private function storeFile($file, ?int $propostaId): string
    {
        $safeId = $propostaId ?: 0;
        $folder = base_path('../public/uploads/propostas/'.$safeId);
        if (!is_dir($folder)) {
            mkdir($folder, 0755, true);
        }

        $extension = $file->getClientOriginalExtension();
        $name = Str::uuid()->toString();
        if ($extension) {
            $name .= '.'.$extension;
        }
        $file->move($folder, $name);

        return 'uploads/propostas/'.$safeId.'/'.$name;
    }

    private function parseEmailList(string $value): array
    {
        $parts = preg_split('/[\s,;]+/', $value);
        $emails = [];
        foreach ($parts as $part) {
            $email = trim((string) $part);
            if ($email === '') {
                continue;
            }
            $email = trim($email, " \t\n\r\0\x0B<>;,");
            $email = rtrim($email, '.');
            if (preg_match('/\.com\.b$/i', $email)) {
                $email .= 'r';
            }
            if ($email !== '') {
                $emails[] = $email;
            }
        }
        return array_values(array_unique($emails));
    }

    private function invalidEmails(array $emails): array
    {
        $invalid = [];
        foreach ($emails as $email) {
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $invalid[] = $email;
            }
        }
        return $invalid;
    }

    private function formatEmailValue($value): string
    {
        if ($value === null) {
            return '-';
        }
        $text = trim((string) $value);
        return $text === '' ? '-' : $text;
    }
}
