<?php

namespace App\Jobs;

use App\Models\Configuracao;
use App\Models\Lead;
use App\Models\LeadNotificationLog;
use App\Models\Usuario;
use App\Services\SmsService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;

class SendLeadAssignedNotifications implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    private const DEFAULT_SMS_TEXTO_LEAD = "Novo lead atribuido: {nome} - {produto} - recebido em {data_hora}. Acesse: {link}";
    private const DEFAULT_WHATSAPP_TEXTO_LEAD = "Novo lead atribuido:\n{nome}\nProduto: {produto}\nRecebido em {data_hora}\nAcesse: {link}";
    private const DEFAULT_EMAIL_ASSUNTO_LEAD = "Voce recebeu um novo lead - {nome}";
    private const DEFAULT_EMAIL_TEXTO_LEAD = "Ola {responsavel},\nVoce recebeu um novo lead:\nCliente: {nome}\nProduto: {produto}\nData/Hora: {data_hora}\nAcesse o lead aqui: {link}\nEquipe Comercial";

    public function __construct(
        private int $leadId,
        private ?int $userId,
        private ?string $assignedAt = null
    ) {
    }

    public function handle(SmsService $smsService): void
    {
        $lead = Lead::query()->find($this->leadId);
        if (!$lead) {
            return;
        }

        $user = $this->userId ? Usuario::query()->find($this->userId) : null;
        if (!$user) {
            $user = $lead->responsavel;
        }
        if (!$user) {
            return;
        }

        $when = $this->assignedAt ? Carbon::parse($this->assignedAt) : now();
        $produto = $lead->tipo_solicitacao ?: ($lead->origem ?: 'Lead');

        $data = [
            'id' => (string) $lead->id,
            'nome' => (string) ($lead->nome ?? ''),
            'produto' => (string) $produto,
            'data' => $when->format('d/m/Y'),
            'hora' => $when->format('H:i'),
            'data_hora' => $when->format('d/m/Y H:i'),
            'link' => $this->buildLeadLink((int) $lead->id),
            'responsavel' => (string) ($user->name ?? ''),
        ];

        $config = Configuracao::query()->first();
        $smsTemplate = $config?->sms_texto_lead ?: env('COMTELE_SMS_TEXTO_LEAD');
        $whatsTemplate = $config?->whatsapp_texto_lead ?: env('WHATSAPP_TEXTO_LEAD');
        $emailSubjectTemplate = $config?->email_assunto_lead ?: env('EMAIL_ASSUNTO_LEAD');
        $emailTextTemplate = $config?->email_texto_lead ?: env('EMAIL_TEXTO_LEAD');

        if (!$smsTemplate) {
            $smsTemplate = self::DEFAULT_SMS_TEXTO_LEAD;
        }
        if (!$whatsTemplate) {
            $whatsTemplate = self::DEFAULT_WHATSAPP_TEXTO_LEAD;
        }
        if (!$emailSubjectTemplate) {
            $emailSubjectTemplate = self::DEFAULT_EMAIL_ASSUNTO_LEAD;
        }
        if (!$emailTextTemplate) {
            $emailTextTemplate = self::DEFAULT_EMAIL_TEXTO_LEAD;
        }

        $smsMessage = $this->renderTemplate($smsTemplate, $data);
        $whatsMessage = $this->renderTemplate($whatsTemplate, $data);
        $emailSubject = $this->renderTemplate($emailSubjectTemplate, $data);
        $emailText = $this->renderTemplate($emailTextTemplate, $data);

        $this->sendSms($smsService, $lead, $user, $smsMessage);
        $this->sendWhatsapp($lead, $user, $whatsMessage);
        $this->sendEmail($lead, $user, $emailSubject, $emailText);
    }

    private function sendSms(SmsService $smsService, Lead $lead, Usuario $user, string $message): void
    {
        $phone = $user->telefone ?? null;
        if (!$phone) {
            $this->logNotification($lead->id, $user->id, 'sms', null, 'failed', $message, 'Telefone nao informado.');
            return;
        }

        $result = $smsService->send($phone, $message);
        $this->logNotification(
            $lead->id,
            $user->id,
            'sms',
            $phone,
            ($result['sent'] ?? false) ? 'sent' : 'failed',
            $message,
            $result['error'] ?? null
        );
    }

    private function sendWhatsapp(Lead $lead, Usuario $user, string $message): void
    {
        $phone = $user->telefone ?? null;
        if (!$phone) {
            $this->logNotification($lead->id, $user->id, 'whatsapp', null, 'failed', $message, 'Telefone nao informado.');
            return;
        }

        $normalized = preg_replace('/\D+/', '', (string) $phone);
        if (!$normalized || strlen($normalized) < 10) {
            $this->logNotification($lead->id, $user->id, 'whatsapp', $phone, 'failed', $message, 'Telefone invalido.');
            return;
        }

        $config = $this->whatsappConfig();
        if (!$config['api_url']) {
            $this->logNotification($lead->id, $user->id, 'whatsapp', $normalized, 'failed', $message, 'API de WhatsApp nao configurada.');
            return;
        }

        $payload = [
            'phone' => $normalized,
            'message' => $message,
        ];
        if ($config['api_key']) {
            $payload['apikey'] = $config['api_key'];
        }

        try {
            $request = Http::timeout(12);
            if ($config['api_token']) {
                $request = $request->withToken($config['api_token']);
            }
            $response = $request->post($config['api_url'], $payload);
        } catch (\Throwable $exception) {
            $this->logNotification($lead->id, $user->id, 'whatsapp', $normalized, 'failed', $message, $exception->getMessage());
            return;
        }

        if (!$response->successful()) {
            $this->logNotification($lead->id, $user->id, 'whatsapp', $normalized, 'failed', $message, 'Falha ao enviar WhatsApp.');
            return;
        }

        $this->logNotification($lead->id, $user->id, 'whatsapp', $normalized, 'sent', $message, null);
    }

    private function sendEmail(Lead $lead, Usuario $user, string $subject, string $text): void
    {
        $email = $user->email ?? null;
        if (!$email) {
            $this->logNotification($lead->id, $user->id, 'email', null, 'failed', $text, 'Email nao informado.');
            return;
        }
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->logNotification($lead->id, $user->id, 'email', $email, 'failed', $text, 'Email invalido.');
            return;
        }

        $fromAddress = config('mail.from.address') ?: env('MAIL_FROM_ADDRESS');
        $fromName = config('mail.from.name') ?: env('MAIL_FROM_NAME');
        if ($fromAddress && !filter_var($fromAddress, FILTER_VALIDATE_EMAIL)) {
            $fromAddress = null;
        }
        if (!$fromAddress) {
            $fromAddress = 'contato@marianalpccapital.com.br';
        }

        $html = nl2br(e($text));

        try {
            Mail::send([], [], function ($message) use ($email, $subject, $fromAddress, $fromName, $html, $text) {
                if ($fromAddress) {
                    $message->from($fromAddress, $fromName ?: null);
                }
                $message->to($email)
                    ->replyTo('contato@marianalpccapital.com.br', 'LPC Capital')
                    ->subject($subject)
                    ->html($html)
                    ->text($text);
            });
        } catch (\Throwable $exception) {
            $this->logNotification($lead->id, $user->id, 'email', $email, 'failed', $text, $exception->getMessage());
            return;
        }

        $this->logNotification($lead->id, $user->id, 'email', $email, 'sent', $text, null);
    }

    private function renderTemplate(string $template, array $data): string
    {
        $replacements = [
            '{id}' => $data['id'] ?? '',
            '{nome}' => $data['nome'] ?? '',
            '{produto}' => $data['produto'] ?? '',
            '{data}' => $data['data'] ?? '',
            '{hora}' => $data['hora'] ?? '',
            '{data_hora}' => $data['data_hora'] ?? '',
            '{link}' => $data['link'] ?? '',
            '{responsavel}' => $data['responsavel'] ?? '',
        ];

        $message = strtr($template, $replacements);
        return trim($message);
    }

    private function buildLeadLink(int $leadId): string
    {
        $base = config('app.url') ?: env('APP_URL', '');
        $base = rtrim($base, '/');
        if (str_ends_with($base, '/api')) {
            $base = substr($base, 0, -4);
        }
        if ($base === '') {
            return '/app/leads/'.$leadId;
        }
        return $base.'/app/leads/'.$leadId;
    }

    private function whatsappConfig(): array
    {
        $config = Configuracao::query()->first();

        return [
            'api_url' => $config?->whatsapp_api_url ?: env('WHATSAPP_API_URL'),
            'api_token' => $config?->whatsapp_api_token ?: env('WHATSAPP_API_TOKEN'),
            'api_key' => $config?->whatsapp_api_key ?: env('WHATSAPP_API_KEY'),
        ];
    }

    private function logNotification(
        int $leadId,
        ?int $userId,
        string $channel,
        ?string $destinatario,
        string $status,
        ?string $message,
        ?string $error
    ): void {
        LeadNotificationLog::query()->create([
            'lead_id' => $leadId,
            'user_id' => $userId,
            'channel' => $channel,
            'destinatario' => $destinatario,
            'status' => $status,
            'message' => $message,
            'error_message' => $error,
        ]);
    }
}
