<?php

namespace App\Services;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class OperacionalService
{
    public function painel(): array
    {
        $tarefas = $this->loadTarefas();
        $checklists = $this->loadChecklists($tarefas);

        return [
            'resumo' => $this->buildResumo($tarefas, $checklists),
            'tarefas' => $tarefas,
            'checklists' => $checklists,
        ];
    }

    private function loadTarefas(): array
    {
        if (!Schema::hasTable('tarefas_operacionais')) {
            return $this->sampleTarefas();
        }

        $tituloColumn = $this->pickColumn('tarefas_operacionais', ['titulo', 'nome', 'tarefa', 'descricao']);
        if (!$tituloColumn) {
            return $this->sampleTarefas();
        }

        $descricaoColumn = $this->pickColumn('tarefas_operacionais', ['descricao', 'detalhe', 'observacao', 'resumo']);
        $statusColumn = $this->pickColumn('tarefas_operacionais', ['status', 'situacao']);
        $prazoColumn = $this->pickColumn('tarefas_operacionais', ['prazo', 'data_prazo', 'data_limite']);
        $etapaColumn = $this->pickColumn('tarefas_operacionais', ['workflow_etapa_id', 'etapa_id']);
        $responsavelColumn = Schema::hasColumn('tarefas_operacionais', 'usuario_id') ? 'usuario_id' : null;

        $query = DB::table('tarefas_operacionais')
            ->select([
                DB::raw('tarefas_operacionais.id as id'),
                $this->selectColumn('tarefas_operacionais', $tituloColumn, 'titulo'),
                $this->selectColumn('tarefas_operacionais', $descricaoColumn, 'descricao'),
                $this->selectColumn('tarefas_operacionais', $statusColumn, 'status'),
                $this->selectColumn('tarefas_operacionais', $prazoColumn, 'prazo'),
                $this->selectColumn('tarefas_operacionais', $etapaColumn, 'etapa_id'),
            ])
            ->orderBy($prazoColumn ?? 'tarefas_operacionais.id');

        if ($responsavelColumn && Schema::hasTable('users') && Schema::hasColumn('users', 'id')) {
            $query->leftJoin('users', "tarefas_operacionais.{$responsavelColumn}", '=', 'users.id');
            $query->addSelect('users.name as responsavel');
        }

        return $query->limit(200)->get()->map(function ($row) {
            return [
                'id' => $row->id ?? null,
                'titulo' => $row->titulo ?? '',
                'descricao' => $row->descricao ?? '',
                'status' => $row->status ?? 'Pendente',
                'responsavel' => $row->responsavel ?? null,
                'prazo' => $row->prazo ?? null,
                'etapa' => $row->etapa_id ? "Etapa {$row->etapa_id}" : null,
            ];
        })->values()->all();
    }

    private function loadChecklists(array $tarefas): array
    {
        if (!Schema::hasTable('checklists_operacionais')) {
            return $this->sampleChecklists($tarefas);
        }

        $tarefaIdColumn = $this->pickColumn('checklists_operacionais', ['tarefa_operacional_id', 'tarefa_id']);
        $statusColumn = $this->pickColumn('checklists_operacionais', ['status', 'situacao']);
        if (!$tarefaIdColumn || !$statusColumn) {
            return $this->sampleChecklists($tarefas);
        }

        $rows = DB::table('checklists_operacionais')
            ->select([
                $this->selectColumn('checklists_operacionais', $tarefaIdColumn, 'tarefa_id'),
                $this->selectColumn('checklists_operacionais', $statusColumn, 'status'),
            ])
            ->get();

        if ($rows->isEmpty()) {
            return $this->sampleChecklists($tarefas);
        }

        $tarefasMap = collect($tarefas)->keyBy('id');

        return $rows->groupBy('tarefa_id')->map(function ($items, $tarefaId) use ($tarefasMap) {
            $total = $items->count();
            $concluidos = $items->filter(function ($item) {
                return $this->contains(strtolower($item->status ?? ''), 'concl');
            })->count();
            $tarefa = $tarefasMap->get($tarefaId);

            return [
                'id' => $tarefaId,
                'tarefa' => $tarefa['titulo'] ?? "Tarefa {$tarefaId}",
                'items_total' => $total,
                'items_concluidos' => $concluidos,
                'status' => $concluidos === $total ? 'Completo' : ($concluidos > 0 ? 'Em andamento' : 'Pendente'),
                'prazo' => $tarefa['prazo'] ?? null,
            ];
        })->values()->all();
    }

    private function buildResumo(array $tarefas, array $checklists): array
    {
        $tarefasAtivas = collect($tarefas)->filter(function ($item) {
            return $this->isActiveStatus($item['status'] ?? '');
        })->count();

        $tarefasPendentes = collect($tarefas)->filter(function ($item) {
            return $this->isPendingStatus($item['status'] ?? '');
        })->count();

        $checklistsPendentes = collect($checklists)->sum(function ($item) {
            $total = $item['items_total'] ?? 0;
            $done = $item['items_concluidos'] ?? 0;
            return max(0, $total - $done);
        });

        $prazos = collect($tarefas)->filter(function ($item) {
            return $this->isDeadlineClose($item['prazo'] ?? null);
        })->count();

        return [
            'tarefas_abertas' => $tarefasAtivas,
            'tarefas_pendentes' => $tarefasPendentes,
            'checklists_pendentes' => $checklistsPendentes,
            'prazos_proximos' => $prazos,
        ];
    }

    private function sampleTarefas(): array
    {
        $now = Carbon::now();

        return [
            [
                'id' => 1,
                'titulo' => 'Conferir documentos de novas propostas',
                'descricao' => 'Validar assinaturas e anexos fiscais antes do envio.',
                'status' => 'Em andamento',
                'responsavel' => 'Ana Ribeiro',
                'prazo' => $now->copy()->addDays(2)->toDateString(),
                'etapa' => 'Analise documental',
            ],
            [
                'id' => 2,
                'titulo' => 'Agendar vistoria dos imoveis',
                'descricao' => 'Sincronizar disponibilidade com clientes e vistoriadores.',
                'status' => 'Pendente',
                'responsavel' => 'Carlos Prado',
                'prazo' => $now->copy()->addDays(4)->toDateString(),
                'etapa' => 'Coordenacao',
            ],
            [
                'id' => 3,
                'titulo' => 'Preparar caderno de documentacao para cartorio',
                'descricao' => 'Reunir certidoes, memoriais e comprovantes de pagamento.',
                'status' => 'Em andamento',
                'responsavel' => 'Lara Monteiro',
                'prazo' => $now->copy()->addDays(6)->toDateString(),
                'etapa' => 'Conferencia final',
            ],
            [
                'id' => 4,
                'titulo' => 'Revisar relatorios de cautela',
                'descricao' => 'Checar acuracia dos dados financeiros antes do envio.',
                'status' => 'Concluido',
                'responsavel' => 'Joaquim Silva',
                'prazo' => $now->copy()->subDays(1)->toDateString(),
                'etapa' => 'Relatorio final',
            ],
        ];
    }

    private function sampleChecklists(array $tarefas = []): array
    {
        $now = Carbon::now();

        return [
            [
                'id' => 1,
                'tarefa' => $this->resolveSampleTitle($tarefas, 0, 'Checklist documental'),
                'items_total' => 6,
                'items_concluidos' => 4,
                'status' => 'Em andamento',
                'prazo' => $now->copy()->addDays(2)->toDateString(),
            ],
            [
                'id' => 2,
                'tarefa' => $this->resolveSampleTitle($tarefas, 1, 'Checklist de vistoria'),
                'items_total' => 8,
                'items_concluidos' => 8,
                'status' => 'Completo',
                'prazo' => $now->copy()->addDays(1)->toDateString(),
            ],
            [
                'id' => 3,
                'tarefa' => $this->resolveSampleTitle($tarefas, 2, 'Checklist de cartorio'),
                'items_total' => 5,
                'items_concluidos' => 1,
                'status' => 'Pendente',
                'prazo' => $now->copy()->addDays(5)->toDateString(),
            ],
        ];
    }

    private function resolveSampleTitle(array $tarefas, int $index, string $fallback): string
    {
        return $tarefas[$index]['titulo'] ?? $fallback;
    }

    private function isActiveStatus(string $value): bool
    {
        $value = strtolower(trim($value));
        return $value === 'em andamento' || $this->contains($value, 'aberta');
    }

    private function isPendingStatus(string $value): bool
    {
        $value = strtolower(trim($value));
        return $value === '' || $this->contains($value, 'pend');
    }

    private function isDeadlineClose($value): bool
    {
        $prazo = $this->parseDate($value);
        if (!$prazo) {
            return false;
        }

        $dias = Carbon::now()->diffInDays($prazo, false);
        return $dias >= 0 && $dias <= 7;
    }

    private function parseDate($value): ?Carbon
    {
        if (!$value) {
            return null;
        }

        try {
            return Carbon::parse($value);
        } catch (\Throwable $exception) {
            return null;
        }
    }

    private function pickColumn(string $table, array $candidates): ?string
    {
        foreach ($candidates as $column) {
            if (Schema::hasColumn($table, $column)) {
                return $column;
            }
        }
        return null;
    }

    private function selectColumn(string $table, ?string $column, string $alias)
    {
        if (!$column) {
            return DB::raw("NULL as {$alias}");
        }
        return DB::raw("{$table}.{$column} as {$alias}");
    }

    private function contains(?string $value, string $needle): bool
    {
        if ($value === null || $value === '') {
            return false;
        }
        return strpos($value, $needle) !== false;
    }
}
