const API_BASE = import.meta.env.VITE_API_BASE_URL || '/adm_capital/api';

function getCookie(name) {
  const value = `; ${document.cookie}`;
  const parts = value.split(`; ${name}=`);
  if (parts.length !== 2) {
    return null;
  }
  return decodeURIComponent(parts.pop().split(';').shift());
}

function buildHeaders() {
  const headers = {
    'Content-Type': 'application/json',
    'X-Requested-With': 'XMLHttpRequest',
  };
  const token = getCookie('XSRF-TOKEN');
  if (token) {
    headers['X-XSRF-TOKEN'] = token;
  }
  return headers;
}

async function request(path, options = {}) {
  const response = await fetch(`${API_BASE}${path}`, {
    credentials: 'include',
    headers: buildHeaders(),
    ...options,
  });

  const contentType = response.headers.get('content-type') || '';
  const isJson = contentType.includes('application/json');

  if (!response.ok) {
    let message = 'Falha ao carregar dados.';
    if (isJson) {
      const payload = await response.json().catch(() => ({}));
      message = payload.message || message;
    } else {
      const text = await response.text().catch(() => '');
      if (text) {
        message = text.slice(0, 180);
      }
    }
    throw new Error(message);
  }

  if (isJson) {
    try {
      return await response.json();
    } catch (error) {
      const text = await response.text().catch(() => '');
      if (text) {
        return { message: text };
      }
      throw new Error('Resposta invalida do servidor.');
    }
  }

  const text = await response.text();
  return { message: text };
}

export function apiGet(path) {
  return request(path);
}

export function apiPost(path, body) {
  return request(path, {
    method: 'POST',
    body: JSON.stringify(body),
  });
}

export function apiPut(path, body) {
  return request(path, {
    method: 'PUT',
    body: JSON.stringify(body),
  });
}

export function apiPatch(path, body) {
  return request(path, {
    method: 'PATCH',
    body: JSON.stringify(body),
  });
}

export function apiDelete(path) {
  return request(path, {
    method: 'DELETE',
  });
}

export function apiUpload(path, formData) {
  const headers = buildHeaders();
  delete headers['Content-Type'];
  return request(path, {
    method: 'POST',
    body: formData,
    headers,
  });
}
