<?php

namespace App\Console\Commands;

use App\Models\AgendaEventoLembrete;
use App\Models\Configuracao;
use App\Services\AgendaService;
use App\Services\SmsService;
use Carbon\Carbon;
use Illuminate\Console\Command;

class EnviarLembretesAgenda extends Command
{
    private const DEFAULT_SMS_TEXTO_AGENDA_USUARIO = "LPC Capital - lembrete de agenda\nOla {responsavel}\nEvento: {titulo}\nInicio: {inicio}\nLocal: {local}\nCliente: {cliente}";
    private const DEFAULT_SMS_TEXTO_AGENDA_CLIENTE = "LPC Capital - lembrete de agenda\nOla {cliente}\nCompromisso: {titulo}\nInicio: {inicio}\nLocal: {local}\nResponsavel: {responsavel}";

    protected $signature = 'agenda:lembretes';

    protected $description = 'Envia lembretes de agenda via SMS';

    public function handle(): int
    {
        $smsService = app(SmsService::class);
        $agendaService = app(AgendaService::class);

        $config = Configuracao::query()->first();
        $templateUsuario = $config?->sms_texto_agenda_usuario
            ?: env('COMTELE_SMS_TEXTO_AGENDA_USUARIO', self::DEFAULT_SMS_TEXTO_AGENDA_USUARIO);
        $templateCliente = $config?->sms_texto_agenda_cliente
            ?: env('COMTELE_SMS_TEXTO_AGENDA_CLIENTE', self::DEFAULT_SMS_TEXTO_AGENDA_CLIENTE);

        $now = Carbon::now();
        $lembretes = AgendaEventoLembrete::query()
            ->where('status', 'pendente')
            ->where('enviar_em', '<=', $now)
            ->with('evento.organizador')
            ->limit(100)
            ->get();

        foreach ($lembretes as $lembrete) {
            if (!$lembrete->destinatario_telefone) {
                $lembrete->status = 'cancelado';
                $lembrete->erro_ultima = 'Telefone nao informado.';
                $lembrete->save();
                continue;
            }

            $evento = $lembrete->evento;
            if (!$evento) {
                $lembrete->status = 'cancelado';
                $lembrete->erro_ultima = 'Evento nao encontrado.';
                $lembrete->save();
                continue;
            }

            $template = $lembrete->destinatario_tipo === 'cliente' ? $templateCliente : $templateUsuario;
            $message = $agendaService->renderSmsTemplate($template, [
                'titulo' => $evento->titulo,
                'inicio' => $evento->inicio_em?->format('d/m/Y H:i'),
                'fim' => $evento->fim_em?->format('d/m/Y H:i'),
                'local' => $evento->localizacao,
                'cliente' => $evento->cliente_nome,
                'responsavel' => $evento->organizador?->name,
            ]);

            $result = $smsService->send($lembrete->destinatario_telefone, $message);
            $lembrete->tentativas = (int) $lembrete->tentativas + 1;

            if ($result['sent']) {
                $lembrete->status = 'enviado';
                $lembrete->enviado_em = Carbon::now();
                $lembrete->erro_ultima = null;
            } else {
                $lembrete->erro_ultima = $result['error'] ?? 'Falha ao enviar.';
                if ($lembrete->tentativas >= 3) {
                    $lembrete->status = 'falhou';
                }
            }

            $lembrete->save();
        }

        $this->info('Lembretes processados: '.$lembretes->count());

        return self::SUCCESS;
    }
}
