<?php

namespace App\Http\Controllers;

use App\Models\Imovel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

class ImovelController extends Controller
{
    public function index()
    {
        $imoveis = Imovel::query()
            ->with('cliente')
            ->orderByDesc('id')
            ->paginate(20);

        return response()->json($imoveis);
    }

    public function create()
    {
        return response()->json(['message' => 'create']);
    }

    public function store(Request $request)
    {
        $registro = Imovel::query()->create($request->only([
            'cliente_id',
            'tipo_imovel_id',
            'cep',
            'logradouro',
            'numero',
            'bairro',
            'cidade',
            'estado',
            'metragem_m2',
            'valor_avaliacao',
            'valor_credito',
            'saldo_devedor',
            'em_condominio',
            'propriedade',
        ]));

        return response()->json($registro->load('cliente'), 201);
    }

    public function sendEmail(Request $request, int $imovelId)
    {
        $payload = $request->validate([
            'email_to' => ['required', 'string', 'max:500'],
        ]);

        $imovel = Imovel::query()->with('cliente')->find($imovelId);
        if (!$imovel) {
            return response()->json(['message' => 'Imovel nao encontrado.'], 404);
        }

        $address = $this->composeAddress($imovel);
        $mapLink = $this->mapLinkForAddress($address);
        $clienteNome = $this->formatEmailValue($imovel->cliente->nome_razao ?? '');
        $subject = 'LPC Capital - Imovel #'.$imovel->id;
        if ($clienteNome !== '-') {
            $subject .= ' - '.$clienteNome;
        }
        $html = $this->buildImovelEmailHtml($imovel, $address, $mapLink);
        $text = $this->buildImovelEmailText($imovel, $address, $mapLink);

        $fromAddress = config('mail.from.address') ?: env('MAIL_FROM_ADDRESS');
        $fromName = config('mail.from.name') ?: env('MAIL_FROM_NAME');

        $emails = $this->parseEmailList($payload['email_to']);
        if (!$emails) {
            return response()->json(['message' => 'Informe ao menos um email valido.'], 422);
        }
        $invalid = $this->invalidEmails($emails);
        if ($invalid) {
            return response()->json([
                'message' => 'Emails invalidos: '.implode(', ', $invalid),
            ], 422);
        }

        try {
            Mail::send([], [], function ($message) use ($emails, $subject, $fromAddress, $fromName, $html, $text) {
                if ($fromAddress) {
                    $message->from($fromAddress, $fromName ?: null);
                }
                $message->to($emails)
                    ->replyTo('contato@marianalpccapital.com.br', 'LPC Capital')
                    ->subject($subject)
                    ->html($html)
                    ->text($text);
            });
        } catch (\Throwable $exception) {
            logger()->error('Falha ao enviar email do imovel.', [
                'error' => $exception->getMessage(),
                'imovel_id' => $imovelId,
                'email_to' => $emails,
            ]);
            return response()->json(['message' => 'Falha ao enviar email.'], 500);
        }

        return response()->json(['message' => 'Email enviado com sucesso.']);
    }

    public function show(int $id)
    {
        $imovel = Imovel::query()
            ->with('cliente')
            ->findOrFail($id);

        return response()->json($imovel);
    }

    public function edit(int $id)
    {
        return response()->json(Imovel::query()->findOrFail($id));
    }

    public function update(Request $request, int $id)
    {
        $registro = Imovel::query()->findOrFail($id);
        $registro->update($request->only([
            'tipo_imovel_id',
            'cep',
            'logradouro',
            'numero',
            'bairro',
            'cidade',
            'estado',
            'metragem_m2',
            'valor_avaliacao',
            'valor_credito',
            'saldo_devedor',
            'em_condominio',
            'propriedade',
        ]));

        return response()->json($registro->load('cliente'));
    }

    public function destroy(int $id)
    {
        Imovel::query()->whereKey($id)->delete();
        return response()->noContent();
    }

    private function composeAddress(Imovel $imovel): string
    {
        $parts = [
            $imovel->logradouro,
            $imovel->numero,
            $imovel->bairro,
            $imovel->cidade,
            $imovel->estado,
            $imovel->cep,
        ];

        $clean = array_values(array_filter(array_map(function ($value) {
            $value = trim((string) $value);
            return $value === '' ? null : $value;
        }, $parts)));

        return implode(', ', $clean);
    }

    private function mapLinkForAddress(string $address): string
    {
        if ($address === '') {
            return '';
        }
        return 'https://www.google.com/maps/search/?api=1&query='.urlencode($address);
    }

    private function formatEmailValue($value): string
    {
        if ($value === null) {
            return '-';
        }
        $text = trim((string) $value);
        return $text === '' ? '-' : $text;
    }

    private function formatEmailMoney($value): string
    {
        if ($value === null || $value === '') {
            return '-';
        }
        if (!is_numeric($value)) {
            return $this->formatEmailValue($value);
        }
        return 'R$ '.number_format((float) $value, 2, ',', '.');
    }

    private function formatEmailBool($value): string
    {
        if ($value === null) {
            return '-';
        }
        if ($value === true || $value === 1 || $value === '1') {
            return 'Sim';
        }
        if ($value === false || $value === 0 || $value === '0') {
            return 'Nao';
        }
        return '-';
    }

    private function tipoImovelLabel($tipoId): string
    {
        $map = [
            1 => 'Apartamento',
            2 => 'Casa',
            3 => 'Sala comercial',
            4 => 'Terreno',
            5 => 'Outros',
        ];
        $normalized = is_numeric($tipoId) ? (int) $tipoId : null;
        return $map[$normalized] ?? $this->formatEmailValue($tipoId);
    }

    private function buildImovelEmailHtml(Imovel $imovel, string $address, string $mapLink): string
    {
        $addressValue = $address !== '' ? $address : '-';
        $mapValue = $mapLink !== '' ? $mapLink : '-';
        $title = $this->formatEmailValue($imovel->id ?? 'Imovel');
        $logoUrl = 'https://marianalpccapital.com.br/img/Logo_LPC_Capital.png';

        $sections = [
            'Cliente' => [
                'ID' => $this->formatEmailValue($imovel->cliente_id ?? ''),
                'Nome' => $this->formatEmailValue($imovel->cliente->nome_razao ?? ''),
                'Tipo' => $this->formatEmailValue($imovel->cliente->tipo_cliente ?? ''),
                'Documento' => $this->formatEmailValue($imovel->cliente->cpf_cnpj ?? ''),
            ],
            'Endereco' => [
                'Endereco' => $addressValue,
                'Mapa' => $mapValue,
            ],
            'Imovel' => [
                'ID' => $this->formatEmailValue($imovel->id ?? ''),
                'Tipo' => $this->tipoImovelLabel($imovel->tipo_imovel_id ?? null),
                'Propriedade' => $this->formatEmailValue($imovel->propriedade ?? ''),
                'Metragem' => $this->formatEmailValue($imovel->metragem_m2 ?? ''),
                'Condominio' => $this->formatEmailBool($imovel->em_condominio ?? null),
                'Valor avaliacao' => $this->formatEmailMoney($imovel->valor_avaliacao ?? null),
                'Valor credito' => $this->formatEmailMoney($imovel->valor_credito ?? null),
                'Saldo devedor' => $this->formatEmailMoney($imovel->saldo_devedor ?? null),
            ],
        ];

        $html = '<!doctype html><html><head><meta charset="utf-8">';
        $html .= '<style>body{font-family:Arial,sans-serif;color:#0f172a;margin:24px}';
        $html .= 'h1{font-size:22px;margin:0}h2{font-size:14px;margin:18px 0 8px;text-transform:uppercase;letter-spacing:.12em;color:#475569}';
        $html .= 'table{width:100%;border-collapse:collapse}th,td{text-align:left;padding:6px 8px;border-bottom:1px solid #e2e8f0;font-size:12px}';
        $html .= 'th{width:32%;color:#475569;font-weight:600}.subtle{color:#64748b;font-size:12px}';
        $html .= '.brand{display:flex;align-items:center;gap:16px;margin-bottom:18px}.brand img{height:36px;width:auto}</style>';
        $html .= '</head><body>';
        $html .= '<div class="brand">';
        if ($logoUrl) {
            $html .= '<img src="'.e($logoUrl).'" alt="LPC Capital" />';
        }
        $html .= '<div><h1>Imovel '.e($title).'</h1>';
        $html .= '<div class="subtle">Gerado em '.e(now()->format('d/m/Y H:i')).'</div></div></div>';

        foreach ($sections as $sectionTitle => $rows) {
            $html .= '<section><h2>'.e($sectionTitle).'</h2><table>';
            foreach ($rows as $label => $value) {
                $html .= '<tr><th>'.e($label).'</th><td>'.e($value).'</td></tr>';
            }
            $html .= '</table></section>';
        }

        $html .= '</body></html>';

        return $html;
    }

    private function buildImovelEmailText(Imovel $imovel, string $address, string $mapLink): string
    {
        $addressValue = $address !== '' ? $address : '-';
        $mapValue = $mapLink !== '' ? $mapLink : '-';

        $lines = [
            'Imovel #'.$this->formatEmailValue($imovel->id ?? ''),
            'Gerado em '.now()->format('d/m/Y H:i'),
            '',
            'Cliente',
            'ID: '.$this->formatEmailValue($imovel->cliente_id ?? ''),
            'Nome: '.$this->formatEmailValue($imovel->cliente->nome_razao ?? ''),
            'Tipo: '.$this->formatEmailValue($imovel->cliente->tipo_cliente ?? ''),
            'Documento: '.$this->formatEmailValue($imovel->cliente->cpf_cnpj ?? ''),
            '',
            'Endereco',
            'Endereco: '.$addressValue,
            'Mapa: '.$mapValue,
            '',
            'Imovel',
            'ID: '.$this->formatEmailValue($imovel->id ?? ''),
            'Tipo: '.$this->tipoImovelLabel($imovel->tipo_imovel_id ?? null),
            'Propriedade: '.$this->formatEmailValue($imovel->propriedade ?? ''),
            'Metragem: '.$this->formatEmailValue($imovel->metragem_m2 ?? ''),
            'Condominio: '.$this->formatEmailBool($imovel->em_condominio ?? null),
            'Valor avaliacao: '.$this->formatEmailMoney($imovel->valor_avaliacao ?? null),
            'Valor credito: '.$this->formatEmailMoney($imovel->valor_credito ?? null),
            'Saldo devedor: '.$this->formatEmailMoney($imovel->saldo_devedor ?? null),
        ];

        return implode("\n", $lines);
    }

    private function parseEmailList(string $value): array
    {
        $parts = preg_split('/[\s,;]+/', $value);
        $emails = [];
        foreach ($parts as $part) {
            $email = trim((string) $part);
            if ($email === '') {
                continue;
            }
            $email = trim($email, " \t\n\r\0\x0B<>;,");
            $email = rtrim($email, '.');
            if ($email !== '') {
                $emails[] = $email;
            }
        }
        return array_values(array_unique($emails));
    }

    private function invalidEmails(array $emails): array
    {
        $invalid = [];
        foreach ($emails as $email) {
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $invalid[] = $email;
            }
        }
        return $invalid;
    }
}
