<?php

namespace App\Http\Controllers;

use App\Models\Cliente;
use App\Models\Configuracao;
use App\Models\Lead;
use App\Models\PessoaFisica;
use App\Models\PessoaJuridica;
use App\Services\LeadService;
use App\Services\SmsService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Schema;

class LeadController extends Controller
{
    private const STATUS = ['novo', 'contato', 'qualificacao', 'proposta', 'convertido', 'perdido'];
    private const DEFAULT_SMS_TEXTO_LEAD = "Novo lead #{id}\nNome: {nome}\nTelefone: {telefone}\nEmail: {email}\nOrigem: {origem}\nTipo: {tipo_solicitacao}\nResponsavel: {responsavel}";
    private const EXTERNAL_SOURCES = [
        'simulacoes_imobiliarias' => [
            'table' => 'simulacoes_imobiliarias',
            'tipo_solicitacao' => 'HOME EQUITY',
        ],
        'simulacoes_consignado' => [
            'table' => 'simulacoes_consignado',
            'tipo_solicitacao' => 'CONSIGNADO',
        ],
    ];
    private const EXTERNAL_DB = 'ipccapitalmarian_lpc_simulacao';
    private const EXTERNAL_DOCUMENTS_TABLE = 'simulacoes_documentos';
    private const TIPO_SOLICITACAO = ['HOME EQUITY', 'CONSIGNADO'];

    public function __construct(private LeadService $service)
    {
    }

    public function index()
    {
        $leads = Lead::query()
            ->with(['responsavel', 'pessoaFisica', 'pessoaJuridica'])
            ->orderByDesc('id')
            ->paginate(20);

        return response()->json($leads);
    }

    public function create()
    {
        return response()->json(['message' => 'create']);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'nome' => ['required', 'string', 'max:255'],
            'email' => ['nullable', 'email', 'max:255'],
            'telefone' => ['nullable', 'string', 'max:30'],
            'origem' => ['nullable', 'string', 'max:120'],
            'tipo_solicitacao' => ['nullable', 'string', 'max:40'],
            'external_source' => ['nullable', 'string', 'max:80'],
            'external_id' => ['nullable', 'integer'],
            'validated_at' => ['nullable', 'date'],
            'status' => ['nullable', 'string', 'in:'.implode(',', self::STATUS)],
            'responsavel_id' => ['nullable', 'integer', 'exists:users,id'],
            'pessoa_fisica_id' => ['nullable', 'integer', 'exists:pessoas_fisicas,cliente_id'],
            'pessoa_juridica_id' => ['nullable', 'integer', 'exists:pessoas_juridicas,cliente_id'],
        ]);

        if (empty($data['status'])) {
            $data['status'] = 'novo';
        }

        $lead = Lead::query()->create($data);
        $this->notifyNewLeadSms($lead);
        return response()->json($lead, 201);
    }

    public function show(int $id)
    {
        $lead = Lead::query()
            ->with(['responsavel', 'pessoaFisica', 'pessoaJuridica'])
            ->findOrFail($id);

        return response()->json($lead);
    }

    public function edit(int $id)
    {
        return response()->json(Lead::query()->findOrFail($id));
    }

    public function update(Request $request, int $id)
    {
        $lead = Lead::query()->findOrFail($id);
        $data = $request->validate([
            'nome' => ['sometimes', 'required', 'string', 'max:255'],
            'email' => ['nullable', 'email', 'max:255'],
            'telefone' => ['nullable', 'string', 'max:30'],
            'origem' => ['nullable', 'string', 'max:120'],
            'tipo_solicitacao' => ['nullable', 'string', 'max:40'],
            'external_source' => ['nullable', 'string', 'max:80'],
            'external_id' => ['nullable', 'integer'],
            'validated_at' => ['nullable', 'date'],
            'responsavel_id' => ['nullable', 'integer', 'exists:users,id'],
            'pessoa_fisica_id' => ['nullable', 'integer', 'exists:pessoas_fisicas,cliente_id'],
            'pessoa_juridica_id' => ['nullable', 'integer', 'exists:pessoas_juridicas,cliente_id'],
        ]);

        $lead->update($data);
        return response()->json($lead);
    }

    public function destroy(int $id)
    {
        Lead::query()->whereKey($id)->delete();
        return response()->noContent();
    }

    public function updateStatus(Request $request, int $id)
    {
        $lead = Lead::query()->findOrFail($id);
        $data = $request->validate([
            'status' => ['required', 'string', 'in:'.implode(',', self::STATUS)],
        ]);

        try {
            $lead = $this->service->transition($lead, $data['status']);
        } catch (\InvalidArgumentException $exception) {
            return response()->json(['message' => $exception->getMessage()], 422);
        }

        return response()->json($lead);
    }

    public function assignResponsavel(Request $request, int $id)
    {
        $lead = Lead::query()->findOrFail($id);
        $data = $request->validate([
            'usuario_id' => ['required', 'integer', 'exists:users,id'],
        ]);

        $lead = $this->service->assignResponsavel($lead, (int) $data['usuario_id']);
        return response()->json($lead);
    }

    public function convertToProposta(int $id)
    {
        $lead = Lead::query()->findOrFail($id);
        $lead = $this->service->convertToProposta($lead);
        return response()->json($lead);
    }

    public function externalIndex()
    {
        $data = collect();

        if (isset(self::EXTERNAL_SOURCES['simulacoes_imobiliarias'])) {
            $source = 'simulacoes_imobiliarias';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $this->externalTable($config['table']);

            try {
                $this->ensureExternalDocumentsTable();
            } catch (\Throwable $exception) {
                logger()->warning('Falha ao garantir tabela de documentos externos.', [
                    'error' => $exception->getMessage(),
                ]);
            }

            try {
                $items = DB::table($table.' as s')
                    ->leftJoin('leads as l', function ($join) use ($source) {
                        $join->on('l.external_id', '=', 's.id')
                            ->where('l.external_source', '=', $source);
                    })
                    ->leftJoin('clientes as c', function ($join) {
                        $join->whereRaw('c.cpf_cnpj COLLATE utf8mb4_unicode_ci = s.cpf COLLATE utf8mb4_unicode_ci');
                    })
                    ->select([
                        's.id as external_id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda',
                        's.cep',
                        's.endereco',
                        's.numero',
                        's.bairro',
                        's.complemento',
                        's.cidade',
                        's.estado',
                        's.valor_imovel',
                        's.tipo_imovel',
                        's.propriedade',
                        's.metragem',
                        's.financiado',
                        's.saldo_financiado',
                        's.condominio',
                        's.valor_emprestimo',
                        's.prazo',
                        's.origem',
                        's.created_at',
                        'l.id as lead_id',
                        'l.tipo_solicitacao',
                        'l.validated_at',
                        'c.id as cliente_id',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $documentsById = $this->loadExternalDocuments($items->pluck('external_id')->filter());

            $data = $data->concat($items->map(function ($item) use ($documentsById, $source, $config) {
                $externalId = (int) $item->external_id;
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $item->lead_id ? (int) $item->lead_id : null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda,
                    'cep' => $item->cep,
                    'endereco' => $item->endereco,
                    'numero' => $item->numero,
                    'bairro' => $item->bairro,
                    'complemento' => $item->complemento,
                    'cidade' => $item->cidade,
                    'estado' => $item->estado,
                    'valor_imovel' => $item->valor_imovel,
                    'tipo_imovel' => $item->tipo_imovel,
                    'propriedade' => $item->propriedade,
                    'metragem' => $item->metragem,
                    'financiado' => $item->financiado,
                    'saldo_financiado' => $item->saldo_financiado,
                    'condominio' => $item->condominio,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => $item->prazo,
                    'origem' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $item->tipo_solicitacao ?: $config['tipo_solicitacao'],
                    'validated_at' => $item->validated_at,
                    'cliente_id' => $item->cliente_id ? (int) $item->cliente_id : null,
                    'documents' => $documentsById[$externalId] ?? [],
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_consignado'])) {
            $source = 'simulacoes_consignado';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $this->externalTable($config['table']);

            try {
                $items = DB::table($table.' as s')
                    ->leftJoin('leads as l', function ($join) use ($source) {
                        $join->on('l.external_id', '=', 's.id')
                            ->where('l.external_source', '=', $source);
                    })
                    ->leftJoin('clientes as c', function ($join) {
                        $join->whereRaw('c.cpf_cnpj COLLATE utf8mb4_unicode_ci = s.cpf COLLATE utf8mb4_unicode_ci');
                    })
                    ->select([
                        's.id as external_id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda_mensal',
                        's.cep',
                        's.endereco',
                        's.numero',
                        's.bairro',
                        's.complemento',
                        's.cidade',
                        's.estado',
                        's.valor_emprestimo',
                        's.data_nascimento',
                        's.situacao_profissional',
                        's.motivo_emprestimo',
                        's.veiculo_nome_cliente',
                        's.placa',
                        's.renavam',
                        's.ano_fabricacao',
                        's.ano_modelo',
                        's.marca',
                        's.modelo',
                        's.pacote_acessorios',
                        's.versao',
                        's.origem',
                        's.created_at',
                        'l.id as lead_id',
                        'l.tipo_solicitacao',
                        'l.validated_at',
                        'c.id as cliente_id',
                    ])
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads externos.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads externos.'], 500);
            }

            $data = $data->concat($items->map(function ($item) use ($source, $config) {
                $externalId = (int) $item->external_id;
                return [
                    'external_id' => $externalId,
                    'external_source' => $source,
                    'lead_id' => $item->lead_id ? (int) $item->lead_id : null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda_mensal,
                    'renda_mensal' => $item->renda_mensal,
                    'cep' => $item->cep,
                    'endereco' => $item->endereco,
                    'numero' => $item->numero,
                    'bairro' => $item->bairro,
                    'complemento' => $item->complemento,
                    'cidade' => $item->cidade,
                    'estado' => $item->estado,
                    'valor_imovel' => null,
                    'tipo_imovel' => 'VEICULO',
                    'propriedade' => null,
                    'metragem' => null,
                    'financiado' => null,
                    'saldo_financiado' => null,
                    'condominio' => null,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $item->tipo_solicitacao ?: $config['tipo_solicitacao'],
                    'validated_at' => $item->validated_at,
                    'cliente_id' => $item->cliente_id ? (int) $item->cliente_id : null,
                    'data_nascimento' => $item->data_nascimento,
                    'situacao_profissional' => $item->situacao_profissional,
                    'motivo_emprestimo' => $item->motivo_emprestimo,
                    'veiculo_nome_cliente' => $item->veiculo_nome_cliente,
                    'placa' => $item->placa,
                    'renavam' => $item->renavam,
                    'ano_fabricacao' => $item->ano_fabricacao,
                    'ano_modelo' => $item->ano_modelo,
                    'marca' => $item->marca,
                    'modelo' => $item->modelo,
                    'pacote_acessorios' => $item->pacote_acessorios,
                    'versao' => $item->versao,
                    'documents' => [],
                ];
            }));
        }

        $data = $data->sortByDesc(function ($item) {
            return $item['created_at'] ?? '';
        })->values();

        return response()->json(['data' => $data]);
    }

    public function availableForPropostas()
    {
        $data = collect();

        if (isset(self::EXTERNAL_SOURCES['simulacoes_imobiliarias'])) {
            $source = 'simulacoes_imobiliarias';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $this->externalTable($config['table']);

            try {
                $items = DB::table($table.' as s')
                    ->leftJoin('leads as l', function ($join) use ($source) {
                        $join->on('l.external_id', '=', 's.id')
                            ->where('l.external_source', '=', $source);
                    })
                    ->leftJoin('clientes as c', function ($join) {
                        $join->whereRaw('c.cpf_cnpj COLLATE utf8mb4_unicode_ci = s.cpf COLLATE utf8mb4_unicode_ci');
                    })
                    ->leftJoin('propostas_credito as p', function ($join) {
                        $join->on('p.cliente_id', '=', 'c.id');
                    })
                    ->select([
                        's.id as external_id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda',
                        's.valor_emprestimo',
                        's.prazo',
                        's.origem',
                        's.created_at',
                        'l.id as lead_id',
                        'l.tipo_solicitacao',
                        'l.validated_at',
                        'c.id as cliente_id',
                    ])
                    ->whereNotNull('c.id')
                    ->whereNull('p.id')
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $data = $data->concat($items->map(function ($item) use ($source, $config) {
                return [
                    'external_id' => (int) $item->external_id,
                    'external_source' => $source,
                    'lead_id' => $item->lead_id ? (int) $item->lead_id : null,
                    'cliente_id' => $item->cliente_id ? (int) $item->cliente_id : null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => $item->prazo,
                    'origem' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $item->tipo_solicitacao ?: $config['tipo_solicitacao'],
                    'validated_at' => $item->validated_at,
                ];
            }));
        }

        if (isset(self::EXTERNAL_SOURCES['simulacoes_consignado'])) {
            $source = 'simulacoes_consignado';
            $config = self::EXTERNAL_SOURCES[$source];
            $table = $this->externalTable($config['table']);

            try {
                $items = DB::table($table.' as s')
                    ->leftJoin('leads as l', function ($join) use ($source) {
                        $join->on('l.external_id', '=', 's.id')
                            ->where('l.external_source', '=', $source);
                    })
                    ->leftJoin('clientes as c', function ($join) {
                        $join->whereRaw('c.cpf_cnpj COLLATE utf8mb4_unicode_ci = s.cpf COLLATE utf8mb4_unicode_ci');
                    })
                    ->leftJoin('propostas_credito as p', function ($join) {
                        $join->on('p.cliente_id', '=', 'c.id');
                    })
                    ->select([
                        's.id as external_id',
                        's.nome',
                        's.email',
                        's.telefone',
                        's.cpf',
                        's.renda_mensal',
                        's.valor_emprestimo',
                        's.origem',
                        's.created_at',
                        'l.id as lead_id',
                        'l.tipo_solicitacao',
                        'l.validated_at',
                        'c.id as cliente_id',
                    ])
                    ->whereNotNull('c.id')
                    ->whereNull('p.id')
                    ->orderByDesc('s.id')
                    ->get();
            } catch (\Throwable $exception) {
                logger()->error('Falha ao carregar leads disponiveis.', [
                    'error' => $exception->getMessage(),
                ]);
                return response()->json(['message' => 'Falha ao carregar leads disponiveis.'], 500);
            }

            $data = $data->concat($items->map(function ($item) use ($source, $config) {
                return [
                    'external_id' => (int) $item->external_id,
                    'external_source' => $source,
                    'lead_id' => $item->lead_id ? (int) $item->lead_id : null,
                    'cliente_id' => $item->cliente_id ? (int) $item->cliente_id : null,
                    'nome' => $item->nome,
                    'email' => $item->email,
                    'telefone' => $item->telefone,
                    'cpf' => $item->cpf,
                    'renda' => $item->renda_mensal,
                    'renda_mensal' => $item->renda_mensal,
                    'valor_emprestimo' => $item->valor_emprestimo,
                    'prazo' => null,
                    'origem' => $item->origem,
                    'created_at' => $item->created_at,
                    'tipo_solicitacao' => $item->tipo_solicitacao ?: $config['tipo_solicitacao'],
                    'validated_at' => $item->validated_at,
                ];
            }));
        }

        $data = $data->sortByDesc(function ($item) {
            return $item['created_at'] ?? '';
        })->values();

        return response()->json(['data' => $data]);
    }

    public function sendExternalEmail(Request $request, int $externalId)
    {
        $payload = $request->validate([
            'email_to' => ['required', 'string', 'max:500'],
            'external_source' => ['nullable', 'string', 'max:80'],
        ]);

        $source = $this->resolveExternalSourceForId($request, $externalId);
        if ($source === null) {
            return response()->json(['message' => 'Lead externo nao encontrado.'], 404);
        }
        if ($source === 'ambiguous') {
            return response()->json(['message' => 'Fonte externa ambigua. Informe external_source.'], 422);
        }
        $config = $this->sourceConfig($source);
        $table = $this->externalTable($config['table']);
        $external = DB::table($table)->where('id', $externalId)->first();
        if (!$external) {
            return response()->json(['message' => 'Lead externo nao encontrado.'], 404);
        }
        $external->renda = $this->externalRenda($external, $source);
        if ($source === 'simulacoes_consignado' && empty($external->tipo_imovel)) {
            $external->tipo_imovel = 'VEICULO';
        }

        $address = $this->composeExternalAddress($external);
        $mapLink = $this->mapLinkForAddress($address);
        $leadId = $external->id ?? $externalId;
        $subjectName = $this->formatEmailValue($external->nome ?? '');
        $subject = 'LPC Capital - Lead #'.$leadId;
        if ($subjectName !== '-') {
            $subject .= ' - '.$subjectName;
        }
        $html = $this->buildLeadEmailHtml($external, $address, $mapLink);
        $text = $this->buildLeadEmailText($external, $address, $mapLink);

        $fromAddress = config('mail.from.address') ?: env('MAIL_FROM_ADDRESS');
        $fromName = config('mail.from.name') ?: env('MAIL_FROM_NAME');

        $emails = $this->parseEmailList($payload['email_to']);
        if (!$emails) {
            return response()->json(['message' => 'Informe ao menos um email valido.'], 422);
        }
        $invalid = $this->invalidEmails($emails);
        if ($invalid) {
            return response()->json([
                'message' => 'Emails invalidos: '.implode(', ', $invalid),
            ], 422);
        }

        try {
            Mail::send([], [], function ($message) use ($emails, $subject, $fromAddress, $fromName, $html, $text) {
                if ($fromAddress) {
                    $message->from($fromAddress, $fromName ?: null);
                }
                $message->to($emails)
                    ->replyTo('contato@marianalpccapital.com.br', 'LPC Capital')
                    ->subject($subject)
                    ->html($html)
                    ->text($text);
            });
        } catch (\Throwable $exception) {
            logger()->error('Falha ao enviar email do lead externo.', [
                'error' => $exception->getMessage(),
                'external_id' => $externalId,
                'email_to' => $emails,
            ]);
            return response()->json(['message' => 'Falha ao enviar email.'], 500);
        }

        return response()->json(['message' => 'Email enviado com sucesso.']);
    }

    public function importExternal(Request $request, int $externalId)
    {
        $payload = $request->validate([
            'tipo_solicitacao' => ['required', 'string', 'in:'.implode(',', self::TIPO_SOLICITACAO)],
            'external_source' => ['nullable', 'string', 'max:80'],
        ]);

        $source = $this->resolveExternalSourceForId($request, $externalId);
        if ($source === null) {
            return response()->json(['message' => 'Lead externo nao encontrado.'], 404);
        }
        if ($source === 'ambiguous') {
            return response()->json(['message' => 'Fonte externa ambigua. Informe external_source.'], 422);
        }
        $config = $this->sourceConfig($source);
        $table = $this->externalTable($config['table']);
        $external = DB::table($table)->where('id', $externalId)->first();
        if (!$external) {
            return response()->json(['message' => 'Lead externo nao encontrado.'], 404);
        }

        $missing = $this->missingExternalFields($external, $source);
        if ($missing) {
            return response()->json([
                'message' => 'Campos obrigatorios ausentes.',
                'fields' => $missing,
            ], 422);
        }

        if (Lead::query()->where('external_source', $source)->where('external_id', $externalId)->exists()) {
            return response()->json(['message' => 'Lead ja importado.'], 409);
        }

        $document = $this->normalizeDocument($external->cpf ?? '');
        if ($document === '') {
            return response()->json(['message' => 'CPF/CNPJ invalido.'], 422);
        }
        $external->renda = $this->externalRenda($external, $source);

        $lead = DB::transaction(function () use ($external, $payload, $document, $source) {
            $clienteId = $this->resolveClienteId($document, $external);
            $tipoCliente = strlen($document) > 11 ? 'PJ' : 'PF';

            if ($tipoCliente === 'PJ') {
                $pessoaId = $this->ensurePessoaJuridica($clienteId, $external, $document);
                $this->ensureEndereco($clienteId, null, $pessoaId, $external);
            } else {
                $pessoaId = $this->ensurePessoaFisica($clienteId, $external);
                $this->ensureEndereco($clienteId, $pessoaId, null, $external);
            }

            if ($source === 'simulacoes_imobiliarias') {
                $this->ensureImovel($clienteId, $external);
            }

            return Lead::query()->create([
                'nome' => $external->nome,
                'email' => $external->email,
                'telefone' => $external->telefone,
                'origem' => $external->origem ?? 'externo',
                'status' => 'novo',
                'responsavel_id' => auth()->id(),
                'pessoa_fisica_id' => $tipoCliente === 'PF' ? $clienteId : null,
                'pessoa_juridica_id' => $tipoCliente === 'PJ' ? $clienteId : null,
                'external_source' => $source,
                'external_id' => $external->id,
                'validated_at' => now(),
                'tipo_solicitacao' => $payload['tipo_solicitacao'],
            ]);
        });

        $this->notifyNewLeadSms($lead);
        return response()->json($lead, 201);
    }

    private function notifyNewLeadSms(Lead $lead): void
    {
        try {
            if (!Schema::hasTable('users') || !Schema::hasColumn('users', 'telefone')) {
                return;
            }

            $users = DB::table('users')
                ->select('id', 'name', 'telefone')
                ->whereNotNull('telefone')
                ->where('telefone', '!=', '')
                ->get();

            if ($users->isEmpty()) {
                return;
            }

            $config = Schema::hasTable('configuracoes') ? Configuracao::query()->first() : null;
            $template = $config?->sms_texto_lead ?: env('COMTELE_SMS_TEXTO_LEAD', self::DEFAULT_SMS_TEXTO_LEAD);
            if (!is_string($template) || trim($template) === '') {
                return;
            }

            $sms = new SmsService();
            $smsConfig = $sms->config();
            if (!$smsConfig['api_url'] || !$smsConfig['auth_key']) {
                return;
            }

            $responsavel = $this->resolveLeadResponsavelNome($lead);
            $message = $this->renderLeadSmsTemplate($template, $lead, $responsavel);
            if ($message === '') {
                return;
            }

            $sentTo = [];
            foreach ($users as $user) {
                $phone = trim((string) ($user->telefone ?? ''));
                $normalized = preg_replace('/\D+/', '', $phone);
                if ($normalized === '' || isset($sentTo[$normalized])) {
                    continue;
                }

                $sentTo[$normalized] = true;
                $result = $sms->send($phone, $message);
                if (!($result['sent'] ?? false)) {
                    logger()->warning('Falha ao enviar SMS do lead.', [
                        'lead_id' => $lead->id,
                        'usuario_id' => $user->id ?? null,
                        'telefone' => $phone,
                        'error' => $result['error'] ?? null,
                    ]);
                }
            }
        } catch (\Throwable $exception) {
            logger()->warning('Falha ao preparar SMS do lead.', [
                'lead_id' => $lead->id ?? null,
                'error' => $exception->getMessage(),
            ]);
        }
    }

    private function resolveLeadResponsavelNome(Lead $lead): string
    {
        if (isset($lead->responsavel) && $lead->responsavel) {
            return trim((string) ($lead->responsavel->name ?? ''));
        }

        if (!Schema::hasTable('users')) {
            return '';
        }

        if ($lead->responsavel_id) {
            $name = DB::table('users')->where('id', $lead->responsavel_id)->value('name');
            return trim((string) ($name ?? ''));
        }

        return '';
    }

    private function renderLeadSmsTemplate(string $template, Lead $lead, string $responsavel = ''): string
    {
        $replacements = [
            '{id}' => (string) $lead->id,
            '{nome}' => trim((string) ($lead->nome ?? '')),
            '{telefone}' => trim((string) ($lead->telefone ?? '')),
            '{email}' => trim((string) ($lead->email ?? '')),
            '{origem}' => trim((string) ($lead->origem ?? '')),
            '{tipo_solicitacao}' => trim((string) ($lead->tipo_solicitacao ?? '')),
            '{responsavel}' => trim($responsavel),
        ];

        $message = strtr($template, $replacements);
        $lines = preg_split('/\r?\n/', $message) ?: [];
        $filtered = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '') {
                continue;
            }
            if (str_ends_with($line, ':')) {
                continue;
            }
            $filtered[] = $line;
        }

        return implode("\n", $filtered);
    }

    private function externalTable(string $table): string
    {
        return self::EXTERNAL_DB.'.'.$table;
    }

    private function resolveExternalSourceForId(Request $request, int $externalId): ?string
    {
        $source = (string) $request->input('external_source', $request->query('external_source', ''));
        if ($source !== '' && array_key_exists($source, self::EXTERNAL_SOURCES)) {
            return $source;
        }

        $matches = $this->detectExternalSources($externalId);
        if (!$matches) {
            return null;
        }
        if (count($matches) > 1) {
            return 'ambiguous';
        }
        return $matches[0];
    }

    private function detectExternalSources(int $externalId): array
    {
        $matches = [];
        foreach (self::EXTERNAL_SOURCES as $key => $config) {
            try {
                $exists = DB::table($this->externalTable($config['table']))
                    ->where('id', $externalId)
                    ->exists();
            } catch (\Throwable $exception) {
                continue;
            }
            if ($exists) {
                $matches[] = $key;
            }
        }
        return $matches;
    }

    private function sourceConfig(string $source): array
    {
        return self::EXTERNAL_SOURCES[$source] ?? self::EXTERNAL_SOURCES['simulacoes_imobiliarias'];
    }

    private function externalRenda(object $external, string $source): float
    {
        if ($source === 'simulacoes_consignado') {
            return (float) ($external->renda_mensal ?? 0);
        }
        return (float) ($external->renda ?? 0);
    }

    private function missingExternalFields(object $external, string $source): array
    {
        if ($source === 'simulacoes_consignado') {
            $required = [
                'nome',
                'email',
                'telefone',
                'cpf',
                'renda_mensal',
                'cep',
                'endereco',
                'numero',
                'bairro',
                'cidade',
                'estado',
                'valor_emprestimo',
                'data_nascimento',
            ];
        } else {
            $required = [
                'nome',
                'email',
                'telefone',
                'cpf',
                'renda',
                'cep',
                'endereco',
                'numero',
                'bairro',
                'cidade',
                'estado',
                'valor_imovel',
                'tipo_imovel',
                'propriedade',
                'metragem',
                'financiado',
                'condominio',
                'valor_emprestimo',
                'prazo',
            ];
        }

        $missing = [];
        foreach ($required as $field) {
            if (!isset($external->$field) || $external->$field === '' || $external->$field === null) {
                $missing[] = $field;
            }
        }

        return $missing;
    }

    private function normalizeDocument(?string $value): string
    {
        return preg_replace('/\D+/', '', (string) $value);
    }

    private function resolveClienteId(string $document, object $external): int
    {
        $cliente = Cliente::query()->where('cpf_cnpj', $document)->first();
        if ($cliente) {
            return (int) $cliente->id;
        }

        $tipoCliente = strlen($document) > 11 ? 'PJ' : 'PF';

        $cliente = Cliente::query()->create([
            'tipo_cliente' => $tipoCliente,
            'nome_razao' => $external->nome,
            'cpf_cnpj' => $document,
            'renda' => $external->renda,
            'usuario_responsavel' => auth()->id(),
            'status' => 'lead',
        ]);

        return (int) $cliente->id;
    }

    private function ensurePessoaFisica(int $clienteId, object $external): int
    {
        $pessoa = PessoaFisica::query()->where('cliente_id', $clienteId)->first();
        if ($pessoa) {
            return (int) $pessoa->cliente_id;
        }

        $pessoa = PessoaFisica::query()->create([
            'cliente_id' => $clienteId,
            'renda_mensal' => $external->renda,
        ]);

        return (int) $pessoa->cliente_id;
    }

    private function ensurePessoaJuridica(int $clienteId, object $external, string $document): int
    {
        $pessoa = PessoaJuridica::query()->where('cliente_id', $clienteId)->first();
        if ($pessoa) {
            return (int) $pessoa->cliente_id;
        }

        $pessoa = PessoaJuridica::query()->create([
            'cliente_id' => $clienteId,
            'cnpj' => $document,
            'renda' => $external->renda,
        ]);

        return (int) $pessoa->cliente_id;
    }

    private function ensureEndereco(int $clienteId, ?int $pessoaFisicaId, ?int $pessoaJuridicaId, object $external): void
    {
        if (!Schema::hasTable('enderecos')) {
            return;
        }

        $data = $this->filterColumns('enderecos', [
            'cliente_id' => $clienteId,
            'logradouro' => $external->endereco ?? null,
            'numero' => $external->numero ?? null,
            'bairro' => $external->bairro ?? null,
            'cidade' => $external->cidade ?? null,
            'uf' => $external->estado ?? null,
            'cep' => $external->cep ?? null,
            'pessoa_fisica_id' => $pessoaFisicaId,
            'pessoa_juridica_id' => $pessoaJuridicaId,
        ], true);

        if (!$data) {
            return;
        }

        $match = [];
        if (isset($data['cliente_id'])) {
            $match['cliente_id'] = $data['cliente_id'];
        }
        if (isset($data['pessoa_fisica_id'])) {
            $match['pessoa_fisica_id'] = $data['pessoa_fisica_id'];
        }
        if (isset($data['pessoa_juridica_id'])) {
            $match['pessoa_juridica_id'] = $data['pessoa_juridica_id'];
        }
        if (isset($data['cep'])) {
            $match['cep'] = $data['cep'];
        }
        if (isset($data['numero'])) {
            $match['numero'] = $data['numero'];
        }

        if ($match) {
            DB::table('enderecos')->updateOrInsert($match, $data);
            return;
        }

        DB::table('enderecos')->insert($data);
    }

    private function ensureImovel(int $clienteId, object $external): ?int
    {
        if (!Schema::hasTable('imoveis')) {
            return null;
        }

        $tipoImovelId = $this->resolveTipoImovelId($external->tipo_imovel ?? null);

        $data = $this->filterColumns('imoveis', [
            'cliente_id' => $clienteId,
            'tipo_imovel_id' => $tipoImovelId,
            'cep' => $external->cep ?? null,
            'logradouro' => $external->endereco ?? null,
            'numero' => $external->numero ?? null,
            'bairro' => $external->bairro ?? null,
            'cidade' => $external->cidade ?? null,
            'estado' => $external->estado ?? null,
            'metragem_m2' => $external->metragem ?? null,
            'valor_avaliacao' => $external->valor_imovel ?? null,
            'valor_credito' => $external->valor_emprestimo ?? null,
            'saldo_devedor' => $external->saldo_financiado ?? null,
            'em_condominio' => $this->parseYesNo($external->condominio ?? null),
            'propriedade' => $external->propriedade ?? null,
        ], true);

        if (!$data) {
            return null;
        }

        $match = [];
        if (isset($data['cliente_id'])) {
            $match['cliente_id'] = $data['cliente_id'];
        }
        if (isset($data['cep'])) {
            $match['cep'] = $data['cep'];
        }
        if (isset($data['numero'])) {
            $match['numero'] = $data['numero'];
        }

        if ($match) {
            $existing = DB::table('imoveis')->where($match)->first();
            if ($existing && isset($existing->id)) {
                return (int) $existing->id;
            }
        }

        return (int) DB::table('imoveis')->insertGetId($data);
    }

    private function parseYesNo(?string $value): ?bool
    {
        $normalized = strtolower(trim((string) $value));
        if ($normalized === '') {
            return null;
        }
        if (in_array($normalized, ['sim', 's', 'yes', 'y', '1'], true)) {
            return true;
        }
        if (in_array($normalized, ['nao', 'n', 'no', '0'], true)) {
            return false;
        }

        return null;
    }

    private function formatEmailValue($value): string
    {
        if ($value === null) {
            return '-';
        }
        $text = trim((string) $value);
        return $text === '' ? '-' : $text;
    }

    private function formatEmailMoney($value): string
    {
        if ($value === null || $value === '') {
            return '-';
        }
        if (!is_numeric($value)) {
            return $this->formatEmailValue($value);
        }
        return 'R$ '.number_format((float) $value, 2, ',', '.');
    }

    private function composeExternalAddress(object $external): string
    {
        $parts = [
            $external->endereco ?? null,
            $external->numero ?? null,
            $external->bairro ?? null,
            $external->cidade ?? null,
            $external->estado ?? null,
            $external->cep ?? null,
        ];

        $clean = array_values(array_filter(array_map(function ($value) {
            $value = trim((string) $value);
            return $value === '' ? null : $value;
        }, $parts)));

        return implode(', ', $clean);
    }

    private function mapLinkForAddress(string $address): string
    {
        if ($address === '') {
            return '';
        }
        return 'https://www.google.com/maps/search/?api=1&query='.urlencode($address);
    }

    private function publicLogoUrl(): string
    {
        return 'https://marianalpccapital.com.br/img/Logo_LPC_Capital.png';
    }

    private function buildLeadEmailHtml(object $external, string $address, string $mapLink): string
    {
        $addressValue = $address !== '' ? $address : '-';
        $mapValue = $mapLink !== '' ? $mapLink : '-';
        $title = $this->formatEmailValue($external->nome ?? 'Lead');
        $logoUrl = $this->publicLogoUrl();

        $sections = [
            'Cliente' => [
                'ID' => $this->formatEmailValue($external->id ?? ''),
                'Nome' => $this->formatEmailValue($external->nome ?? ''),
                'Email' => $this->formatEmailValue($external->email ?? ''),
                'Telefone' => $this->formatEmailValue($external->telefone ?? ''),
                'CPF' => $this->formatEmailValue($external->cpf ?? ''),
                'Renda' => $this->formatEmailMoney($external->renda ?? null),
                'Origem' => $this->formatEmailValue($external->origem ?? ''),
            ],
            'Endereco' => [
                'Endereco' => $addressValue,
                'Complemento' => $this->formatEmailValue($external->complemento ?? ''),
                'Cidade/UF' => $this->formatEmailValue($external->cidade ?? '').' / '.$this->formatEmailValue($external->estado ?? ''),
                'Mapa' => $mapValue,
            ],
            'Imovel' => [
                'Tipo' => $this->formatEmailValue($external->tipo_imovel ?? ''),
                'Propriedade' => $this->formatEmailValue($external->propriedade ?? ''),
                'Metragem' => $this->formatEmailValue($external->metragem ?? ''),
                'Valor' => $this->formatEmailMoney($external->valor_imovel ?? null),
                'Financiado' => $this->formatEmailValue($external->financiado ?? ''),
                'Saldo' => $this->formatEmailMoney($external->saldo_financiado ?? null),
                'Condominio' => $this->formatEmailValue($external->condominio ?? ''),
            ],
            'Emprestimo' => [
                'Valor solicitado' => $this->formatEmailMoney($external->valor_emprestimo ?? null),
                'Prazo' => isset($external->prazo) ? $this->formatEmailValue($external->prazo).' meses' : '-',
                'Tipo solicitacao' => $this->formatEmailValue($external->tipo_solicitacao ?? ''),
            ],
        ];

        $html = '<!doctype html><html><head><meta charset="utf-8">';
        $html .= '<style>';
        $html .= 'body{font-family:Arial,sans-serif;color:#0f172a;margin:24px;}';
        $html .= 'h1{font-size:22px;margin:0;}';
        $html .= 'h2{font-size:13px;margin:18px 0 8px;text-transform:uppercase;letter-spacing:0.12em;color:#475569;}';
        $html .= 'table{width:100%;border-collapse:collapse;}';
        $html .= 'th,td{text-align:left;padding:6px 8px;border-bottom:1px solid #e2e8f0;font-size:12px;}';
        $html .= 'th{width:32%;color:#475569;font-weight:600;}';
        $html .= '.subtle{color:#64748b;font-size:12px;}';
        $html .= '.brand{display:flex;align-items:center;gap:16px;margin-bottom:18px;}';
        $html .= '.brand img{height:36px;width:auto;}';
        $html .= '</style></head><body>';
        $html .= '<div class="brand">';
        if ($logoUrl !== '') {
            $html .= '<img src="'.e($logoUrl).'" alt="LPC Capital" />';
        }
        $html .= '<div><h1>Lead '.e($title).'</h1>';
        $html .= '<div class="subtle">Gerado em '.e(now()->format('d/m/Y H:i')).'</div></div></div>';

        foreach ($sections as $sectionTitle => $rows) {
            $html .= '<section><h2>'.e($sectionTitle).'</h2><table>';
            foreach ($rows as $label => $value) {
                $html .= '<tr><th>'.e($label).'</th><td>'.e($value).'</td></tr>';
            }
            $html .= '</table></section>';
        }

        $html .= '</body></html>';

        return $html;
    }

    private function buildLeadEmailText(object $external, string $address, string $mapLink): string
    {
        $addressValue = $address !== '' ? $address : '-';
        $mapValue = $mapLink !== '' ? $mapLink : '-';

        $lines = [
            'Lead '.($this->formatEmailValue($external->nome ?? '')),
            'Gerado em '.now()->format('d/m/Y H:i'),
            '',
            'Cliente',
            'ID: '.$this->formatEmailValue($external->id ?? ''),
            'Nome: '.$this->formatEmailValue($external->nome ?? ''),
            'Email: '.$this->formatEmailValue($external->email ?? ''),
            'Telefone: '.$this->formatEmailValue($external->telefone ?? ''),
            'CPF: '.$this->formatEmailValue($external->cpf ?? ''),
            'Renda: '.$this->formatEmailMoney($external->renda ?? null),
            'Origem: '.$this->formatEmailValue($external->origem ?? ''),
            '',
            'Endereco',
            'Endereco: '.$addressValue,
            'Complemento: '.$this->formatEmailValue($external->complemento ?? ''),
            'Cidade/UF: '.$this->formatEmailValue($external->cidade ?? '').' / '.$this->formatEmailValue($external->estado ?? ''),
            'Mapa: '.$mapValue,
            '',
            'Imovel',
            'Tipo: '.$this->formatEmailValue($external->tipo_imovel ?? ''),
            'Propriedade: '.$this->formatEmailValue($external->propriedade ?? ''),
            'Metragem: '.$this->formatEmailValue($external->metragem ?? ''),
            'Valor: '.$this->formatEmailMoney($external->valor_imovel ?? null),
            'Financiado: '.$this->formatEmailValue($external->financiado ?? ''),
            'Saldo: '.$this->formatEmailMoney($external->saldo_financiado ?? null),
            'Condominio: '.$this->formatEmailValue($external->condominio ?? ''),
            '',
            'Emprestimo',
            'Valor solicitado: '.$this->formatEmailMoney($external->valor_emprestimo ?? null),
            'Prazo: '.(isset($external->prazo) ? $this->formatEmailValue($external->prazo).' meses' : '-'),
            'Tipo solicitacao: '.$this->formatEmailValue($external->tipo_solicitacao ?? ''),
        ];

        return implode("\n", $lines);
    }

    private function parseEmailList(string $value): array
    {
        $parts = preg_split('/[\s,;]+/', $value);
        $emails = [];
        foreach ($parts as $part) {
            $email = trim((string) $part);
            if ($email === '') {
                continue;
            }
            $email = trim($email, " \t\n\r\0\x0B<>;,");
            $email = rtrim($email, '.');
            if ($email !== '') {
                $emails[] = $email;
            }
        }
        return array_values(array_unique($emails));
    }

    private function invalidEmails(array $emails): array
    {
        $invalid = [];
        foreach ($emails as $email) {
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $invalid[] = $email;
            }
        }
        return $invalid;
    }

    private function resolveTipoImovelId(?string $tipo): ?int
    {
        $normalized = strtolower(trim((string) $tipo));
        if ($normalized === '') {
            return null;
        }

        if (Schema::hasTable('tipo_imoveis')) {
            $columns = Schema::getColumnListing('tipo_imoveis');
            $nameColumn = null;
            foreach (['nome', 'descricao', 'titulo'] as $candidate) {
                if (in_array($candidate, $columns, true)) {
                    $nameColumn = $candidate;
                    break;
                }
            }
            if ($nameColumn) {
                $row = DB::table('tipo_imoveis')
                    ->whereRaw('LOWER('.$nameColumn.') = ?', [$normalized])
                    ->first();
                if ($row && isset($row->id)) {
                    return (int) $row->id;
                }
            }
        }

        $map = [
            'apartamento' => 1,
            'casa' => 2,
            'sala comercial' => 3,
            'terreno' => 4,
            'outros' => 5,
        ];

        return $map[$normalized] ?? null;
    }

    private function filterColumns(string $table, array $data, bool $dropEmpty = false): array
    {
        $columns = Schema::getColumnListing($table);
        $filtered = [];
        foreach ($data as $key => $value) {
            if (!in_array($key, $columns, true)) {
                continue;
            }
            if ($dropEmpty && ($value === null || $value === '')) {
                continue;
            }
            $filtered[$key] = $value;
        }

        return $filtered;
    }

    private function ensureExternalDocumentsTable(): void
    {
        $table = $this->externalTable(self::EXTERNAL_DOCUMENTS_TABLE);
        $sql = <<<'SQL'
CREATE TABLE IF NOT EXISTS %s (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    simulacao_id BIGINT UNSIGNED NOT NULL,
    arquivo VARCHAR(255) NOT NULL,
    tipo VARCHAR(60) NULL,
    created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    INDEX simulacoes_documentos_simulacao_id_index (simulacao_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
SQL;

        DB::statement(sprintf($sql, $table));
    }

    private function loadExternalDocuments($externalIds): array
    {
        if ($externalIds->isEmpty()) {
            return [];
        }

        try {
            $rows = DB::table($this->externalTable(self::EXTERNAL_DOCUMENTS_TABLE))
                ->whereIn('simulacao_id', $externalIds->values()->all())
                ->orderBy('id')
                ->get();
        } catch (\Throwable $exception) {
            return [];
        }

        $documents = [];
        foreach ($rows as $row) {
            $documents[$row->simulacao_id][] = [
                'id' => $row->id,
                'arquivo' => $row->arquivo,
                'tipo' => $row->tipo,
            ];
        }

        return $documents;
    }
}
