<?php

namespace App\Http\Controllers;

use App\Models\Cliente;
use App\Models\PessoaFisica;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PessoaFisicaController extends Controller
{
    public function index()
    {
        $pessoas = PessoaFisica::query()
            ->with('cliente')
            ->orderByDesc('cliente_id')
            ->paginate(20);

        return response()->json($pessoas);
    }

    public function create()
    {
        return response()->json(['message' => 'create']);
    }

    public function store(Request $request)
    {
        $clienteId = $this->resolveCliente($request, 'PF');

        $registro = PessoaFisica::query()->create(array_merge(
            $request->only([
                'profissao',
                'renda_mensal',
                'conjuge_nome',
                'conjuge_data_nascimento',
                'conjuge_estado_civil',
            ]),
            ['cliente_id' => $clienteId]
        ));

        $this->storeEndereco($clienteId, $request);

        return response()->json($registro->load('cliente'), 201);
    }

    public function show(int $id)
    {
        $pessoa = PessoaFisica::query()
            ->with('cliente')
            ->findOrFail($id);

        return response()->json($pessoa);
    }

    public function edit(int $id)
    {
        return response()->json(PessoaFisica::query()->findOrFail($id));
    }

    public function update(Request $request, int $id)
    {
        $registro = PessoaFisica::query()->findOrFail($id);
        $registro->update($request->only([
            'profissao',
            'renda_mensal',
            'conjuge_nome',
            'conjuge_data_nascimento',
            'conjuge_estado_civil',
        ]));

        $clienteData = $request->only([
            'nome_razao',
            'cpf_cnpj',
            'data_nascimento',
            'estado_civil',
            'renda',
            'usuario_responsavel',
            'status',
        ]);
        if (array_filter($clienteData)) {
            $registro->cliente()->update($clienteData);
        }

        return response()->json($registro->load('cliente'));
    }

    public function destroy(int $id)
    {
        PessoaFisica::query()->whereKey($id)->delete();
        return response()->noContent();
    }

    private function resolveCliente(Request $request, string $tipo): int
    {
        $clienteId = $request->input('cliente_id');
        if ($clienteId) {
            return (int) $clienteId;
        }

        $payload = $request->only([
            'nome_razao',
            'cpf_cnpj',
            'data_nascimento',
            'estado_civil',
            'renda',
            'usuario_responsavel',
            'status',
        ]);

        if (empty($payload['nome_razao']) && $request->filled('nome')) {
            $payload['nome_razao'] = $request->input('nome');
        }
        if (empty($payload['cpf_cnpj']) && $request->filled('cpf')) {
            $payload['cpf_cnpj'] = $request->input('cpf');
        }

        $payload['tipo_cliente'] = $tipo;

        $cliente = Cliente::query()->create($payload);

        return (int) $cliente->id;
    }

    private function storeEndereco(int $clienteId, Request $request): void
    {
        $payload = [
            'cliente_id' => $clienteId,
            'tipo_endereco' => 'RESIDENCIAL',
            'cep' => $request->input('cep'),
            'logradouro' => $request->input('endereco'),
            'numero' => $request->input('numero'),
            'bairro' => $request->input('bairro'),
            'cidade' => $request->input('cidade'),
            'estado' => $request->input('estado'),
            'complemento' => $request->input('complemento'),
        ];

        $hasData = false;
        foreach ($payload as $key => $value) {
            if ($key === 'cliente_id' || $key === 'tipo_endereco') {
                continue;
            }
            if ($value !== null && $value !== '') {
                $hasData = true;
                break;
            }
        }
        if (!$hasData) {
            return;
        }

        DB::table('enderecos')->insert($payload);
    }
}
