<?php

namespace App\Services;

use Illuminate\Support\Carbon;

class JuridicoService
{
    public function painel(): array
    {
        $processos = $this->sampleProcessos();
        $documentos = $this->sampleDocumentos();
        $pendencias = $this->samplePendencias();

        return [
            'widgets' => $this->buildWidgets($processos, $documentos, $pendencias),
            'processos' => $processos,
            'documentos' => $documentos,
            'pendencias' => $pendencias,
        ];
    }

    private function buildWidgets(array $processos, array $documentos, array $pendencias): array
    {
        $processosAtivos = collect($processos)->filter(function ($item) {
            return ($item['status'] ?? '') !== 'Concluido';
        })->count();
        $processosCriticos = collect($processos)->filter(function ($item) {
            return ($item['tone'] ?? '') === 'rose';
        })->count();

        $pendenciasAbertas = collect($pendencias)->filter(function ($item) {
            return strtolower(trim($item['status'] ?? '')) !== 'resolvida';
        })->count();
        $pendenciasAltas = collect($pendencias)->filter(function ($item) {
            return strtolower(trim($item['prioridade'] ?? '')) === 'alta';
        })->count();

        $documentosPendentes = collect($documentos)->filter(function ($item) {
            return strtolower(trim($item['status'] ?? '')) !== 'recebido';
        })->count();
        $documentosUrgentes = collect($documentos)->filter(function ($item) {
            return !empty($item['urgencia']);
        })->count();

        $certidoesPendentes = collect($documentos)->filter(function ($item) {
            $documento = strtolower($item['documento'] ?? '');
            $status = strtolower(trim($item['status'] ?? ''));
            return $this->contains($documento, 'certida') && $status !== 'recebido';
        })->count();

        return [
            [
                'label' => 'Processos ativos',
                'value' => $processosAtivos,
                'note' => "{$processosCriticos} criticos na fila",
                'tone' => $processosCriticos ? 'rose' : 'emerald',
            ],
            [
                'label' => 'Pendencias abertas',
                'value' => $pendenciasAbertas,
                'note' => "{$pendenciasAltas} prioridade alta",
                'tone' => $pendenciasAltas ? 'amber' : 'emerald',
            ],
            [
                'label' => 'Documentos pendentes',
                'value' => $documentosPendentes,
                'note' => "{$documentosUrgentes} vencendo esta semana",
                'tone' => $documentosUrgentes ? 'amber' : 'emerald',
            ],
            [
                'label' => 'Certidoes pendentes',
                'value' => $certidoesPendentes,
                'note' => 'Cartorios ativos',
                'tone' => $certidoesPendentes ? 'sky' : 'emerald',
            ],
        ];
    }

    private function sampleProcessos(): array
    {
        $now = Carbon::now();

        return [
            [
                'id' => 'PJ-1024',
                'contrato' => 'CT-1009',
                'cliente' => 'Maria Lopes',
                'tipo' => 'Execucao de garantia',
                'etapa' => 'Intimacao inicial',
                'responsavel' => 'Dra. Souza',
                'status' => 'Em andamento',
                'prazo' => $now->copy()->addDays(12)->format('Y-m-d'),
                'tone' => $this->toneFromStatus('Em andamento'),
            ],
            [
                'id' => 'PJ-1025',
                'contrato' => 'CT-1042',
                'cliente' => 'Grupo Horizonte',
                'tipo' => 'Revisao contratual',
                'etapa' => 'Coleta de provas',
                'responsavel' => 'Dr. Bruno',
                'status' => 'Aguardando documentos',
                'prazo' => $now->copy()->addDays(7)->format('Y-m-d'),
                'tone' => $this->toneFromStatus('Aguardando documentos'),
            ],
            [
                'id' => 'PJ-1026',
                'contrato' => 'CT-0981',
                'cliente' => 'Joao Martins',
                'tipo' => 'Penhora',
                'etapa' => 'Despacho do juiz',
                'responsavel' => 'Dra. Lima',
                'status' => 'Critico',
                'prazo' => $now->copy()->addDays(3)->format('Y-m-d'),
                'tone' => $this->toneFromStatus('Critico'),
            ],
            [
                'id' => 'PJ-1027',
                'contrato' => 'CT-0920',
                'cliente' => 'Empresa Solaris',
                'tipo' => 'Distrato',
                'etapa' => 'Arquivado',
                'responsavel' => 'Equipe interna',
                'status' => 'Concluido',
                'prazo' => $now->copy()->subDays(10)->format('Y-m-d'),
                'tone' => $this->toneFromStatus('Concluido'),
            ],
        ];
    }

    private function sampleDocumentos(): array
    {
        $now = Carbon::now();

        return [
            [
                'id' => 1,
                'processo' => 'PJ-1024',
                'documento' => 'Contrato original',
                'origem' => 'Cliente',
                'status' => 'Recebido',
                'prazo' => $now->copy()->subDays(5)->format('Y-m-d'),
                'urgencia' => false,
                'tone' => $this->toneFromStatus('Recebido'),
            ],
            [
                'id' => 2,
                'processo' => 'PJ-1025',
                'documento' => 'Certidao de onus',
                'origem' => 'Cartorio 12 - SP',
                'status' => 'Pendente',
                'prazo' => $now->copy()->addDays(4)->format('Y-m-d'),
                'urgencia' => true,
                'tone' => $this->toneFromStatus('Pendente'),
            ],
            [
                'id' => 3,
                'processo' => 'PJ-1026',
                'documento' => 'Laudo pericial',
                'origem' => 'Pericia externa',
                'status' => 'Emissao',
                'prazo' => $now->copy()->addDays(6)->format('Y-m-d'),
                'urgencia' => true,
                'tone' => $this->toneFromStatus('Emissao'),
            ],
            [
                'id' => 4,
                'processo' => 'PJ-1027',
                'documento' => 'Termo de acordo',
                'origem' => 'Advocacia interna',
                'status' => 'Recebido',
                'prazo' => $now->copy()->subDays(30)->format('Y-m-d'),
                'urgencia' => false,
                'tone' => $this->toneFromStatus('Recebido'),
            ],
        ];
    }

    private function samplePendencias(): array
    {
        $now = Carbon::now();

        return [
            [
                'id' => 201,
                'contrato' => 'CT-1009',
                'pendencia' => 'Certidao negativa de debitos',
                'detalhe' => 'Emitir no cartorio central.',
                'origem' => 'Cartorio 12 - SP',
                'status' => 'Aberta',
                'prioridade' => 'Alta',
                'prazo' => $now->copy()->addDays(5)->format('Y-m-d'),
                'tone' => $this->toneFromStatus('Aberta'),
            ],
            [
                'id' => 202,
                'contrato' => 'CT-1042',
                'pendencia' => 'Atualizar procuracao',
                'detalhe' => 'Nova assinatura do cliente.',
                'origem' => 'Cliente',
                'status' => 'Em revisao',
                'prioridade' => 'Media',
                'prazo' => $now->copy()->addDays(8)->format('Y-m-d'),
                'tone' => $this->toneFromStatus('Em revisao'),
            ],
            [
                'id' => 203,
                'contrato' => 'CT-0981',
                'pendencia' => 'Averbar penhora',
                'detalhe' => 'Confirmar custas e protocolo.',
                'origem' => 'Cartorio 3 - RJ',
                'status' => 'Aberta',
                'prioridade' => 'Alta',
                'prazo' => $now->copy()->addDays(2)->format('Y-m-d'),
                'tone' => $this->toneFromStatus('Aberta'),
            ],
            [
                'id' => 204,
                'contrato' => 'CT-0920',
                'pendencia' => 'Enviar comprovante',
                'detalhe' => 'Registrar comprovante final.',
                'origem' => 'Equipe interna',
                'status' => 'Resolvida',
                'prioridade' => 'Baixa',
                'prazo' => $now->copy()->subDays(10)->format('Y-m-d'),
                'tone' => $this->toneFromStatus('Resolvida'),
            ],
        ];
    }

    private function toneFromStatus(?string $status): string
    {
        $value = strtolower(trim($status ?? ''));

        if ($this->contains($value, 'crit') || $this->contains($value, 'urg')) {
            return 'rose';
        }

        if ($this->contains($value, 'pend') || $this->contains($value, 'aberta') || $this->contains($value, 'rev')) {
            return 'amber';
        }

        if ($this->contains($value, 'concl') || $this->contains($value, 'resol')) {
            return 'emerald';
        }

        if ($this->contains($value, 'receb') || $this->contains($value, 'completo')) {
            return 'emerald';
        }

        if ($this->contains($value, 'aguard')) {
            return 'sky';
        }

        return 'slate';
    }

    private function contains(?string $value, string $needle): bool
    {
        if ($value === null || $value === '') {
            return false;
        }
        return strpos($value, $needle) !== false;
    }
}
